#!/usr/bin/env python3
"""
Crowd Source Python Client

Connects to the Socket.IO server to receive accelerometer data from phones
and send control messages back to phones in a specific room.

Usage:
    python client.py --room org-slug/room-slug
    python client.py --room org-slug/room-slug --server http://VPS_IP:3107
"""

import argparse
import sys
import signal
import time
from datetime import datetime
import socketio
import threading

# Global variables
sio = socketio.Client()
running = True
message_count = 0


def format_timestamp(timestamp_ms):
    """Format timestamp to readable format"""
    dt = datetime.fromtimestamp(timestamp_ms / 1000)
    return dt.strftime('%H:%M:%S.%f')[:-3]


def format_accelerometer_data(data):
    """Format accelerometer data for display"""
    return (
        f"Phone: {data['phoneId'][:8]}... | "
        f"X: {data['data']['x']:7.2f} | "
        f"Y: {data['data']['y']:7.2f} | "
        f"Z: {data['data']['z']:7.2f} | "
        f"Time: {format_timestamp(data['timestamp'])}"
    )


@sio.event
def connect():
    """Handle connection to server"""
    print("[✓] Connected to Socket.IO server")
    print(f"[i] Session ID: {sio.sid}")


@sio.event
def disconnect():
    """Handle disconnection from server"""
    print("[✗] Disconnected from server")


@sio.event
def connect_error(data):
    """Handle connection error"""
    print(f"[✗] Connection error: {data}")


@sio.event
def error(data):
    """Handle server error"""
    print(f"[✗] Server error: {data.get('message', 'Unknown error')}")


@sio.event
def room_status(data):
    """Handle room status update"""
    print(f"\n[i] Room Status: {data['room']}")
    print(f"    Phones connected: {data['phoneCount']}")
    print(f"    Time: {format_timestamp(data['timestamp'])}")


@sio.event
def phone_joined(data):
    """Handle phone joined event"""
    print(f"\n[+] Phone joined: {data['phoneId'][:8]}...")


@sio.event
def phone_left(data):
    """Handle phone left event"""
    print(f"\n[-] Phone left: {data['phoneId'][:8]}...")


@sio.event
def accelerometer_data(data):
    """Handle incoming accelerometer data"""
    global message_count
    message_count += 1

    # Clear line and print data
    print(f"\r[{message_count:4d}] {format_accelerometer_data(data)}", end='', flush=True)


def send_control_message(message_type, message_text):
    """Send a control message to all phones in the room"""
    try:
        payload = {
            'type': message_type,
            'message': message_text,
            'timestamp': int(time.time() * 1000)
        }
        sio.emit('control_message', payload)
        print(f"\n[→] Sent control message: {message_type} - {message_text}")
    except Exception as e:
        print(f"\n[✗] Error sending control message: {e}")


def input_thread():
    """Background thread to handle user input for sending control messages"""
    global running

    print("\n" + "="*70)
    print("Commands:")
    print("  Type a message and press Enter to send an alert to all phones")
    print("  Type 'quit' or 'exit' to disconnect")
    print("="*70 + "\n")

    while running:
        try:
            user_input = input()

            if user_input.lower() in ['quit', 'exit', 'q']:
                print("\n[i] Disconnecting...")
                running = False
                sio.disconnect()
                break
            elif user_input.strip():
                send_control_message('alert', user_input.strip())
        except EOFError:
            # Handle Ctrl+D
            break
        except Exception as e:
            print(f"\n[✗] Input error: {e}")


def signal_handler(sig, frame):
    """Handle Ctrl+C gracefully"""
    global running
    print("\n\n[i] Received interrupt signal, shutting down...")
    running = False
    sio.disconnect()
    sys.exit(0)


def main():
    parser = argparse.ArgumentParser(
        description='Crowd Source Python Client - Receive accelerometer data from phones',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog='''
Examples:
  python client.py --room myorg/room1
  python client.py --room myorg/room1 --server http://192.168.1.100:3107
        '''
    )

    parser.add_argument(
        '--room',
        type=str,
        required=True,
        help='Room to connect to (format: org-slug/room-slug)'
    )

    parser.add_argument(
        '--server',
        type=str,
        default='http://localhost:3107',
        help='Socket.IO server URL (default: http://localhost:3107)'
    )

    args = parser.parse_args()

    # Validate room format
    if '/' not in args.room:
        print("[✗] Error: Room must be in format 'org-slug/room-slug'")
        sys.exit(1)

    # Set up signal handler for graceful shutdown
    signal.signal(signal.SIGINT, signal_handler)

    print("\n" + "="*70)
    print("Crowd Source Python Client")
    print("="*70)
    print(f"Server: {args.server}")
    print(f"Room:   {args.room}")
    print("="*70 + "\n")

    try:
        # Connect to server
        print("[i] Connecting to server...")
        sio.connect(args.server, transports=['websocket', 'polling'])

        # Join room as Python client
        print(f"[i] Joining room: {args.room}")
        sio.emit('join_as_python_client', {'room': args.room})

        # Start input thread for sending control messages
        input_handler = threading.Thread(target=input_thread, daemon=True)
        input_handler.start()

        # Keep the main thread alive
        while running:
            sio.sleep(0.1)

    except socketio.exceptions.ConnectionError as e:
        print(f"\n[✗] Failed to connect to server: {e}")
        print(f"[i] Make sure the Socket.IO server is running at {args.server}")
        sys.exit(1)
    except Exception as e:
        print(f"\n[✗] Unexpected error: {e}")
        sys.exit(1)
    finally:
        if sio.connected:
            sio.disconnect()
        print(f"\n[i] Total messages received: {message_count}")
        print("[i] Client shut down")


if __name__ == '__main__':
    main()
