#!/usr/bin/env python3
"""
Crowd Source Python Client - Textual UI

Modern Terminal UI for receiving accelerometer data from phones
and sending control messages back to phones in a specific room.

Usage:
    python client_ui.py --room org-slug/room-slug --server https://ws.vec4.net
"""

import argparse
import sys
import threading
import time
import gc
import tracemalloc
from pathlib import Path

# Add parent directory to path for imports
sys.path.insert(0, str(Path(__file__).parent))

from textual.app import App, ComposeResult
from textual.containers import Container, Vertical, VerticalScroll
from textual.widgets import Header, Footer, Static, RichLog
from textual.binding import Binding
from textual.reactive import reactive

from utils.socket_client import SocketClient
from utils.osc_service import OSCService
from utils.osc_validators import rgb_to_hex
from widgets.status_bar import StatusBar
from widgets.tabbed_container import TabbedContainer
from widgets.controls_tab import ControlsTab
from widgets.config_tab import ConfigTab
from widgets.event_log_tab import EventLogTab
from widgets.latency_diagnostic_tab import LatencyDiagnosticTab


class CrowdSourceApp(App):
    """Crowd Source Textual UI Application"""

    # App-level state
    connected_phones: reactive[list] = reactive([])
    selected_phone_ids: reactive[set] = reactive(set())
    phone_index_map: reactive[dict] = reactive({})
    interaction_counts: reactive[dict] = reactive({})
    error_message: reactive[str] = reactive("")
    target_mode: reactive[str] = reactive("all")  # "all" or "selected"

    # OSC configuration
    osc_enabled: reactive[bool] = reactive(False)
    osc_use_device_index: reactive[bool] = reactive(False)
    osc_send_host: reactive[str] = reactive("127.0.0.1")
    osc_send_port: reactive[int] = reactive(7000)
    osc_receive_port: reactive[int] = reactive(7001)

    CSS = """
    Screen {
        background: $surface;
    }

    #status-bar {
        dock: top;
        height: 3;
        background: $primary-darken-2;
        padding: 1;
    }

    .section-title {
        text-style: bold;
        color: $accent;
        margin-bottom: 1;
        margin-top: 1;
    }

    .subsection-label {
        color: $text-muted;
        margin-top: 1;
        margin-bottom: 0;
    }

    .nav-buttons {
        layout: horizontal;
        height: auto;
        margin-bottom: 1;
    }

    .nav-buttons Button {
        width: 1fr;
        margin-right: 1;
    }

    #frequency-controls {
        layout: horizontal;
        height: auto;
        margin-bottom: 1;
    }

    #frequency-input {
        width: 10;
        margin-right: 1;
    }

    #frequency-unit {
        width: 4;
        content-align: center middle;
        margin-right: 1;
    }

    #apply-freq-btn {
        min-width: 10;
    }

    #preset-label {
        margin-top: 1;
        color: $text-muted;
    }

    #frequency-presets {
        layout: horizontal;
        height: auto;
        margin-bottom: 1;
    }

    #frequency-presets Button {
        margin-right: 1;
    }

    #alert-controls {
        layout: horizontal;
        height: auto;
        margin-bottom: 1;
    }

    #alert-input {
        width: 1fr;
        margin-right: 1;
    }

    #send-alert-btn {
        min-width: 10;
    }

    #target-radio {
        margin-bottom: 1;
    }

    #phone-controls {
        layout: horizontal;
        height: auto;
        margin-top: 1;
        margin-bottom: 1;
    }

    #phone-controls Button {
        width: 1fr;
        margin-right: 1;
    }

    #phones-container {
        min-height: 10;
        border: solid $primary;
        padding: 1;
        margin-top: 1;
        margin-bottom: 1;
    }

    .phone-item {
        padding: 0 1;
        height: auto;
    }

    .phone-item:hover {
        background: $primary-darken-1;
        text-style: bold;
    }

    .empty-state {
        color: $text-muted;
        text-align: center;
        padding: 2;
    }

    .config-section {
        border: solid $primary;
        padding: 1;
        margin-bottom: 2;
        height: auto;
    }

    .config-section Static {
        height: auto;
        margin-top: 0;
        margin-bottom: 0;
    }

    .config-section Input {
        width: 100%;
        margin-bottom: 1;
    }

    .config-action-buttons {
        layout: horizontal;
        height: auto;
        margin-top: 1;
    }

    .config-action-buttons Button {
        width: 1fr;
        margin-right: 1;
    }

    .mode-buttons {
        layout: horizontal;
        height: auto;
        margin-bottom: 1;
    }

    .mode-buttons Button {
        width: 1fr;
        margin-right: 1;
    }

    .multibtn-button-item {
        border: solid $accent-darken-2;
        padding: 1;
        margin-bottom: 1;
        height: auto;
    }

    .multibtn-button-item Static {
        height: auto;
    }

    .multibtn-button-item Input {
        width: 100%;
        margin-bottom: 1;
    }

    .multibtn-button-item Checkbox {
        margin-bottom: 1;
    }

    .multibtn-button-row {
        layout: horizontal;
        height: auto;
        margin-bottom: 1;
        align: left middle;
    }

    .multibtn-button-row Checkbox {
        width: auto;
        margin-right: 1;
    }

    .multibtn-button-row Input {
        width: 1fr;
    }

    /* Index assignment mode selector */
    .mode-selector-container {
        height: auto;
        margin-bottom: 1;
    }

    .mode-description {
        height: auto;
        margin-bottom: 2;
        padding: 1;
        background: $panel;
        border: solid $primary;
    }

    /* History controls */
    #history-controls {
        height: auto;
        margin-bottom: 1;
    }

    #history-controls.hidden {
        display: none;
    }

    #history-count {
        padding-left: 1;
        content-align: center middle;
    }

    .info-text {
        color: $text-muted;
    }

    /* Management controls */
    #management-controls {
        height: auto;
        margin-bottom: 1;
        margin-top: 1;
    }
    """

    BINDINGS = [
        Binding("q", "quit", "Quit", show=True),
        Binding("ctrl+c", "quit", "Quit", show=False),
    ]

    def __init__(self, room: str, server_url: str):
        super().__init__()
        self.room = room
        self.server_url = server_url
        self.socket_client = None
        self.osc_service = None
        self.connection_thread = None

        # UI throttling state (for performance - currently disabled)
        self.last_ui_update_time = 0
        self.last_ui_update_count = 0
        self.ui_update_interval_ms = 100  # Update UI every 100ms
        self.ui_update_every_n = 10       # Or every 10th message (whichever comes first)

    def compose(self) -> ComposeResult:
        """Create child widgets"""
        yield Header()
        yield StatusBar(id="status-bar")
        yield TabbedContainer()
        yield Footer()

    def on_mount(self) -> None:
        """Initialize the application"""
        self.title = "Crowd Source Client"
        self.sub_title = f"Room: {self.room}"

        # CRITICAL: Disable automatic garbage collection to prevent OSC stream pauses
        # GC pauses can cause 500-1000ms gaps in sensor data streaming
        gc.disable()
        gc.set_threshold(0, 0, 0)  # Disable all generation thresholds

        # Update status bar
        status_bar = self.query_one(StatusBar)
        status_bar.room = self.room

        # Initialize socket client
        self.socket_client = SocketClient(self.server_url, self.room)

        # Set up callbacks
        self.socket_client.set_on_connect(self.on_socket_connect)
        self.socket_client.set_on_disconnect(self.on_socket_disconnect)
        self.socket_client.set_on_data(self.on_socket_data)
        self.socket_client.set_on_error(self.on_socket_error)
        self.socket_client.set_on_phone_joined(self.on_phone_joined)
        self.socket_client.set_on_phone_left(self.on_phone_left)
        self.socket_client.set_on_interaction(self.on_interaction)
        self.socket_client.set_on_clock_sync(self.on_clock_sync)

        # Connect in background thread
        self.connection_thread = threading.Thread(
            target=self.socket_client.connect, daemon=True
        )
        self.connection_thread.start()

        # Initialize OSC service
        self.osc_service = OSCService(self)
        self.osc_service.start(self.osc_receive_port)

        # CRITICAL: NO LOGGING - Every log message creates Rich text objects
        # which causes 1000+ allocations through Textual's rendering pipeline

        # Disable automatic GC (no logging)
        # Install minimal GC callback for critical errors only
        def gc_callback(phase, info):
            """Only log VERY expensive collections (>100ms)"""
            if phase == 'stop':
                duration_ms = info.get('duration', 0) * 1000
                if duration_ms > 100:  # Only log if VERY slow
                    # Print to stdout instead of Events tab (lighter)
                    print(f"[GC WARNING] Gen-{info.get('generation')} took {duration_ms:.1f}ms")

        gc.callbacks.append(gc_callback)

        # Schedule manual GC during idle time (not in sensor pipeline)
        self.set_interval(5.0, self._manual_gc_collect)

        # DISABLED: Profiling creates 500+ objects per interval
        # tracemalloc.start()
        # self._tracemalloc_snapshot = None
        # self.set_interval(10.0, self._profile_allocations)

    def on_socket_connect(self) -> None:
        """Handle socket connection"""
        self.call_from_thread(self._update_connection_status, True)

        # Show which transport is being used (for Windows latency diagnostics)
        transport = self.socket_client.sio.transport()

        event_log_tab = self.query_one("#event-log-tab", EventLogTab)
        self.call_from_thread(
            event_log_tab.write,
            f"[green]✓ Connected to server via {transport}[/green]"
        )

    def on_clock_sync(self, offset: float) -> None:
        """Handle clock synchronization completion"""
        event_log_tab = self.query_one("#event-log-tab", EventLogTab)
        self.call_from_thread(
            event_log_tab.write,
            f"[cyan]⏱ Clock synchronized (offset: {offset:.2f}ms)[/cyan]"
        )

        # Update status bar with clock offset (always visible)
        status_bar = self.query_one(StatusBar)
        self.call_from_thread(self._update_clock_offset, offset)

        # Update latency diagnostic tab with the offset
        latency_diagnostic_tab = self.query_one("#latency-diagnostic-tab", LatencyDiagnosticTab)
        self.call_from_thread(
            latency_diagnostic_tab.set_clock_offset,
            offset
        )

    def on_socket_disconnect(self) -> None:
        """Handle socket disconnection"""
        self.call_from_thread(self._update_connection_status, False)

        # Clear all state on disconnect (it's now invalid)
        # State will be automatically rebuilt on reconnect when server sends phone_joined events
        self.call_from_thread(self._clear_state_on_disconnect)

        event_log_tab = self.query_one("#event-log-tab", EventLogTab)
        self.call_from_thread(
            event_log_tab.write, "[red]✗ Disconnected from server[/red]"
        )

    def on_socket_data(self, data: dict) -> None:
        """
        Handle incoming sensor data.

        CRITICAL: Minimal processing in hot path to prevent object allocation.
        OSC is sent FIRST from socket thread for minimal latency.
        """
        # Send OSC immediately - NO OTHER PROCESSING!
        if self.osc_enabled and self.osc_service:
            phone_id = data.get('phoneId')
            phone_index = self.phone_index_map.get(phone_id)
            self.osc_service.send_sensor_data(data, phone_index)

    def on_socket_error(self, error_msg: str) -> None:
        """Handle socket errors"""
        event_log_tab = self.query_one("#event-log-tab", EventLogTab)
        self.call_from_thread(
            event_log_tab.write, f"[red]✗ Error: {error_msg}[/red]"
        )

    def _manual_gc_collect(self) -> None:
        """
        Manually trigger garbage collection during idle time.

        Called every 5 seconds via set_interval. Only collects generation 0
        (youngest objects) which is fast (~1-5ms) and prevents expensive
        generation 2 collections that would pause the sensor pipeline.

        This is safe to call from the UI thread since it's quick.
        """
        # Just collect - NO LOGGING (logging creates Rich text objects)
        gc.collect(0)

    def _profile_allocations(self) -> None:
        """
        Profile memory allocations to identify allocation hotspots.

        DISABLED: Profiling itself creates 500+ objects per interval.
        We've already identified the allocations are from Textual/Rich.
        """
        pass

    def _update_connection_status(self, connected: bool) -> None:
        """Update connection status in UI (called from thread)"""
        status_bar = self.query_one(StatusBar)
        status_bar.is_connected = connected

    def _update_clock_offset(self, offset: float) -> None:
        """Update clock offset in UI (called from thread)"""
        status_bar = self.query_one(StatusBar)
        status_bar.clock_offset = offset

    def _should_update_ui(self, count: int) -> bool:
        """
        Determine if UI should be updated (throttling logic).

        Updates when either:
        - ui_update_interval_ms has elapsed since last update
        - ui_update_every_n messages have been received

        Returns:
            bool: True if UI should be updated
        """
        now = time.time() * 1000
        time_since_last = now - self.last_ui_update_time
        count_since_last = count - self.last_ui_update_count

        # Update if either threshold is met
        if time_since_last >= self.ui_update_interval_ms or count_since_last >= self.ui_update_every_n:
            self.last_ui_update_time = now
            self.last_ui_update_count = count
            return True
        return False


    def _assign_phone_index(self, phone_id: str) -> int:
        """
        Assign lowest available index to phone.

        Finds the smallest unused index starting from 1.
        Indices are derived from current phone_index_map to ensure synchronization.

        Args:
            phone_id: Phone identifier

        Returns:
            int: The assigned index (lowest available)
        """
        # Derive used indices from current phone_index_map (always in sync)
        used_indices = set(self.phone_index_map.values())

        # Find lowest available index starting from 1
        index = 1
        while index in used_indices:
            index += 1

        return index

    def on_phone_joined(self, data: dict) -> None:
        """Handle phone joined event"""
        self.call_from_thread(self._on_phone_joined, data)

    def _on_phone_joined(self, data: dict) -> None:
        """Handle phone joined (on main thread)"""
        import time
        phone_id = data.get('phoneId')
        timestamp = data.get('timestamp', time.time() * 1000)

        # Defensive check: If phone already exists, remove old entry first
        # This handles race conditions where phone rejoins before disconnect is processed
        if phone_id in self.phone_index_map:
            event_log_tab = self.query_one("#event-log-tab", EventLogTab)
            old_index = self.phone_index_map[phone_id]
            event_log_tab.write(
                f"[yellow]⚠ Phone {phone_id[:8]} rejoining (was #{old_index}). Cleaning up old entry.[/yellow]"
            )
            self._on_phone_left({'phoneId': phone_id, 'timestamp': timestamp})

        # Assign index using lowest available algorithm
        index = self._assign_phone_index(phone_id)

        # Update phone index map (use reactive pattern)
        phone_index_map = dict(self.phone_index_map)
        phone_index_map[phone_id] = index
        self.phone_index_map = phone_index_map

        # Add to connected phones list
        phone_info = {
            'phone_id': phone_id,
            'index': index,
            'connected_at': timestamp / 1000,
            'socket_id': ''
        }
        phones = list(self.connected_phones)
        phones.append(phone_info)
        self.connected_phones = phones

        # Forward to OSC if enabled
        if self.osc_enabled and self.osc_service:
            self.osc_service.send_phone_joined(
                phone_id,
                index,
                int(timestamp),
                len(self.connected_phones)
            )

        # Log
        event_log_tab = self.query_one("#event-log-tab", EventLogTab)
        event_log_tab.write(f"[green]✓ Phone #{index}: {phone_id[:8]} joined[/green]")

    def on_phone_left(self, data: dict) -> None:
        """Handle phone left event"""
        self.call_from_thread(self._on_phone_left, data)

    def _on_phone_left(self, data: dict) -> None:
        """Handle phone left (on main thread)"""
        import time
        phone_id = data.get('phoneId')
        timestamp = data.get('timestamp', time.time() * 1000)
        index = self.phone_index_map.get(phone_id, '?')

        # Remove from connected phones
        phones = [p for p in self.connected_phones if p['phone_id'] != phone_id]
        self.connected_phones = phones

        # Remove from phone_index_map (use reactive pattern)
        if phone_id in self.phone_index_map:
            phone_index_map = dict(self.phone_index_map)
            del phone_index_map[phone_id]
            self.phone_index_map = phone_index_map  # Properly trigger reactive update

        # Note: No need to manually free indices - they're derived from phone_index_map
        # When we remove from phone_index_map above, the index automatically becomes available

        # Remove from selection if selected
        if phone_id in self.selected_phone_ids:
            selected = set(self.selected_phone_ids)
            selected.discard(phone_id)
            self.selected_phone_ids = selected

        # Forward to OSC if enabled
        if self.osc_enabled and self.osc_service:
            self.osc_service.send_phone_left(
                phone_id,
                index,
                int(timestamp),
                len(self.connected_phones)
            )

        # Log
        event_log_tab = self.query_one("#event-log-tab", EventLogTab)
        event_log_tab.write(f"[red]✗ Phone #{index}: {phone_id[:8]} left[/red]")

    def _clear_state_on_disconnect(self) -> None:
        """
        Clear phone state when disconnected from server.

        When disconnected, our local state becomes invalid since we're no longer
        receiving real-time updates. Clear everything so we start fresh on reconnect.

        The socket server will automatically send phone_joined events for all
        existing phones when we reconnect, rebuilding our state cleanly.
        """
        # Clear phone tracking
        self.connected_phones = []
        self.phone_index_map = {}
        self.selected_phone_ids = set()
        self.interaction_counts = {}

        # Clear OSC aggregator state if enabled
        if self.osc_enabled and self.osc_service:
            self.osc_service.aggregator.reset()

        # Log for visibility
        event_log_tab = self.query_one("#event-log-tab", EventLogTab)
        event_log_tab.write("[blue]ℹ Cleared local state (disconnected, will resync on reconnect)[/blue]")

    def on_interaction(self, data: dict) -> None:
        """Handle interaction event"""
        self.call_from_thread(self._on_interaction, data)

    def _on_interaction(self, data: dict) -> None:
        """Handle interaction (on main thread)"""
        phone_id = data.get('phoneId', 'unknown')
        page = data.get('page', '?')
        action = data.get('action', '?')
        timestamp = data.get('timestamp', 0)
        button_index = data.get('buttonIndex')
        button_text = data.get('buttonText')

        # Update interaction count
        counts = dict(self.interaction_counts)
        counts[phone_id] = counts.get(phone_id, 0) + 1
        self.interaction_counts = counts

        # Get phone index
        index = self.phone_index_map.get(phone_id, '?')
        count = counts[phone_id]

        # Format timestamp
        from datetime import datetime
        dt = datetime.fromtimestamp(timestamp / 1000)
        time_str = dt.strftime('%H:%M:%S')

        # Build interaction detail
        if button_index is not None and button_text:
            detail = f"{action} [Button {button_index}: \"{button_text}\"]"
        else:
            detail = action

        # Forward to OSC if enabled
        if self.osc_enabled and self.osc_service:
            phone_index = self.phone_index_map.get(phone_id)
            self.osc_service.send_interaction(data, phone_index)

        # Log to event log
        event_log_tab = self.query_one("#event-log-tab", EventLogTab)
        event_log_tab.write(
            f"[yellow]{time_str}[/yellow] ← [cyan]#{index}: {phone_id[:8]}[/cyan] "
            f"{detail} on [magenta]{page}[/magenta] (total: {count})"
        )

    def on_controls_tab_navigate_to_page(self, message: ControlsTab.NavigateToPage) -> None:
        """Handle page navigation request"""
        event_log_tab = self.query_one("#event-log-tab", EventLogTab)

        success = self.socket_client.navigate_to_page(
            page=message.page,
            phone_id=message.phone_ids,
            config=message.config
        )

        if success:
            if message.phone_ids:
                target = f"{len(message.phone_ids)} phone(s)"
            else:
                target = "all phones"
            config_str = " with config" if message.config else ""
            event_log_tab.write(
                f"[blue]→ Navigated {target} to {message.page}{config_str}[/blue]"
            )
        else:
            event_log_tab.write(
                "[red]✗ Failed to send navigation command (not connected)[/red]"
            )

    def on_config_tab_navigate_to_page(self, message: ConfigTab.NavigateToPage) -> None:
        """Handle page navigation request from Config tab"""
        event_log_tab = self.query_one("#event-log-tab", EventLogTab)

        success = self.socket_client.navigate_to_page(
            page=message.page,
            phone_id=message.phone_ids,
            config=message.config
        )

        if success:
            if message.phone_ids:
                target = f"{len(message.phone_ids)} phone(s)"
            else:
                target = "all phones"
            config_str = " with config" if message.config else ""
            event_log_tab.write(
                f"[blue]→ Navigated {target} to {message.page}{config_str}[/blue]"
            )
        else:
            event_log_tab.write(
                "[red]✗ Failed to send navigation command (not connected)[/red]"
            )

    def on_config_tab_configure_page(self, message: ConfigTab.ConfigurePage) -> None:
        """Handle page configuration request from Config tab"""
        event_log_tab = self.query_one("#event-log-tab", EventLogTab)

        success = self.socket_client.configure_page(
            page=message.page,
            config=message.config,
            phone_id=message.phone_ids
        )

        if success:
            if message.phone_ids:
                target = f"{len(message.phone_ids)} phone(s)"
            else:
                target = "all phones"
            event_log_tab.write(
                f"[blue]→ Configured {message.page} for {target}[/blue]"
            )
        else:
            event_log_tab.write(
                "[red]✗ Failed to send configuration (not connected)[/red]"
            )

    def on_controls_tab_send_alert(self, message: ControlsTab.SendAlert) -> None:
        """Handle send alert message"""
        event_log_tab = self.query_one("#event-log-tab", EventLogTab)
        if self.socket_client.send_control_message("alert", message.message):
            event_log_tab.write(
                f"[blue]→ Sent alert: {message.message}[/blue]"
            )
        else:
            event_log_tab.write(
                "[red]✗ Failed to send alert (not connected)[/red]"
            )

    def on_controls_tab_set_frequency(self, message: ControlsTab.SetFrequency) -> None:
        """Handle frequency change request"""
        event_log_tab = self.query_one("#event-log-tab", EventLogTab)

        # Send set_frequency event to server
        if self.socket_client and self.socket_client.is_connected:
            self.socket_client.sio.emit('set_frequency', {
                'room': self.room,
                'frequency': message.frequency_ms
            })

            # Calculate Hz for display
            frequency_hz = 1000 / message.frequency_ms
            event_log_tab.write(
                f"[blue]→ Set frequency: {message.frequency_ms}ms ({frequency_hz:.1f} Hz)[/blue]"
            )
        else:
            event_log_tab.write(
                "[red]✗ Failed to set frequency (not connected)[/red]"
            )

    def handle_osc_color(self, device_identifier, r: float, g: float, b: float) -> None:
        """
        Handle OSC color command.

        Args:
            device_identifier: Target phone ID (str) or index (int), None for all phones
            r, g, b: RGB values (0.0-1.0)
        """
        from utils.osc_validators import resolve_device_identifier

        # Resolve identifier to phone ID
        phone_id = None
        if device_identifier is not None:
            phone_id = resolve_device_identifier(device_identifier, dict(self.phone_index_map))
            if not phone_id:
                # Invalid identifier, log and return
                event_log_tab = self.query_one("#event-log-tab", EventLogTab)
                event_log_tab.write(f"[red]✗ OSC: Invalid device identifier {device_identifier}[/red]")
                return

        hex_color = rgb_to_hex(r, g, b)
        config = {'color': hex_color}

        if phone_id:
            self.socket_client.configure_page('pageColour', config, phone_id=[phone_id])
        else:
            self.socket_client.configure_page('pageColour', config)

        # Log with original identifier
        event_log_tab = self.query_one("#event-log-tab", EventLogTab)
        target = f"device {device_identifier}" if device_identifier is not None else "all phones"
        event_log_tab.write(f"[blue]→ OSC: Set color for {target} to {hex_color}[/blue]")

    def handle_osc_navigate(self, device_identifier, page_path: str) -> None:
        """
        Handle OSC navigate command.

        Args:
            device_identifier: Target phone ID (str) or index (int), None for all phones
            page_path: Page to navigate to
        """
        import uuid
        from utils.osc_validators import resolve_device_identifier

        # Resolve identifier to phone ID
        phone_id = None
        if device_identifier is not None:
            phone_id = resolve_device_identifier(device_identifier, dict(self.phone_index_map))
            if not phone_id:
                # Invalid identifier, log and return
                event_log_tab = self.query_one("#event-log-tab", EventLogTab)
                event_log_tab.write(f"[red]✗ OSC: Invalid device identifier {device_identifier}[/red]")
                return

        # Generate page instance ID for OSC tracking
        page_instance_id = str(uuid.uuid4())

        if phone_id:
            self.socket_client.navigate_to_page(page_path, phone_id=[phone_id])
        else:
            self.socket_client.navigate_to_page(page_path)

        # Track page change for OSC metadata (with empty config for now)
        if self.osc_service:
            self.osc_service.send_page_change(page_path, page_instance_id, {})

        # Log with original identifier
        event_log_tab = self.query_one("#event-log-tab", EventLogTab)
        target = f"device {device_identifier}" if device_identifier is not None else "all phones"
        event_log_tab.write(f"[blue]→ OSC: Navigate {target} to {page_path}[/blue]")

    def handle_osc_alert(self, device_identifier, message: str) -> None:
        """
        Handle OSC alert command.

        Args:
            device_identifier: Target phone ID (str) or index (int), None for all phones
            message: Alert message text
        """
        from utils.osc_validators import resolve_device_identifier

        # Resolve identifier to phone ID
        phone_id = None
        if device_identifier is not None:
            phone_id = resolve_device_identifier(device_identifier, dict(self.phone_index_map))
            if not phone_id:
                # Invalid identifier, log and return
                event_log_tab = self.query_one("#event-log-tab", EventLogTab)
                event_log_tab.write(f"[red]✗ OSC: Invalid device identifier {device_identifier}[/red]")
                return

        # Note: send_control_message doesn't support phone_id targeting
        # This sends to all phones regardless of device_identifier parameter
        self.socket_client.send_control_message('alert', message)

        # Log with original identifier
        event_log_tab = self.query_one("#event-log-tab", EventLogTab)
        target = f"device {device_identifier}" if device_identifier is not None else "all phones"
        event_log_tab.write(f"[blue]→ OSC: Alert to {target}: {message}[/blue]")

    def handle_osc_frequency(self, ms: int) -> None:
        """
        Handle OSC frequency command.

        Args:
            ms: Frequency in milliseconds
        """
        if self.socket_client and self.socket_client.is_connected:
            self.socket_client.sio.emit('set_frequency', {
                'room': self.room,
                'frequency': ms
            })

            # Log
            event_log_tab = self.query_one("#event-log-tab", EventLogTab)
            hz = 1000 / ms
            event_log_tab.write(f"[blue]→ OSC: Set frequency to {ms}ms ({hz:.1f} Hz)[/blue]")

    def action_quit(self) -> None:
        """Quit the application"""
        if self.osc_service:
            self.osc_service.stop()
        if self.socket_client:
            self.socket_client.disconnect()
        self.exit()


def main():
    """Main entry point"""
    parser = argparse.ArgumentParser(
        description="Crowd Source Python Client - Terminal UI",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  python client_ui.py --room myorg/room1 --server https://ws.vec4.net
  python client_ui.py --room test-org/test-room --server https://ws.vec4.net
        """,
    )

    parser.add_argument(
        "--room",
        type=str,
        required=True,
        help="Room to connect to (format: org-slug/room-slug)",
    )

    parser.add_argument(
        "--server",
        type=str,
        default="https://ws.vec4.net",
        help="Socket.IO server URL (default: https://ws.vec4.net)",
    )

    args = parser.parse_args()

    # Validate room format
    if "/" not in args.room:
        print("Error: Room must be in format 'org-slug/room-slug'")
        sys.exit(1)

    # Run the app
    app = CrowdSourceApp(room=args.room, server_url=args.server)
    app.run()


if __name__ == "__main__":
    main()
