#!/usr/bin/env python3
"""
Standalone example: Flatten nested sensor data from WebSocket

This script connects to the crowd-source WebSocket server, joins a demo room,
and flattens the nested sensor data structure for easier database storage or CSV export.

Transformation:
  data: {
    "accelerometer": {"x": 0.15, "y": -0.23, "z": 9.81},
    "gyroscope": {"alpha": 45.2, "beta": -12.8, "gamma": 3.5}
  }

Becomes:
  data: {
    "accelerometer_x": 0.15,
    "accelerometer_y": -0.23,
    "accelerometer_z": 9.81,
    "gyroscope_alpha": 45.2,
    "gyroscope_beta": -12.8,
    "gyroscope_gamma": 3.5
  }

Usage:
  python flatten_example.py
"""

import socketio
import json
import os


# Configuration
SERVER_URL = os.getenv('SOCKET_SERVER_URL', 'https://ws.vec4.net')
ROOM = 'demo-org/demo-room'


def flatten_sensor_data(data):
    """
    Flatten the nested sensor data structure.

    Converts nested sensor objects into flat key-value pairs with underscore naming.
    Example: data["accelerometer"]["x"] -> data["accelerometer_x"]

    Args:
        data: Original sensor data payload with nested structure

    Returns:
        dict: Flattened payload with same metadata but flattened data object
    """
    # Extract the nested data object
    nested_data = data.get('data', {})

    # Create flattened data object
    flattened_data = {}

    # Flatten each sensor type
    for sensor_type, sensor_values in nested_data.items():
        if isinstance(sensor_values, dict):
            # Flatten nested keys: accelerometer.x -> accelerometer_x
            for key, value in sensor_values.items():
                flattened_key = f"{sensor_type}_{key}"
                flattened_data[flattened_key] = value

    # Build new payload with flattened data
    flattened_payload = {
        'type': data.get('type'),
        'phoneId': data.get('phoneId'),
        'room': data.get('room'),
        'timestamp': data.get('timestamp'),
        'data': flattened_data
    }

    return flattened_payload


def main():
    """Main function to connect and stream flattened sensor data"""

    # Create Socket.IO client
    sio = socketio.Client()

    # Connection event handler
    @sio.event
    def connect():
        print(f"[Connected] Connected to {SERVER_URL}")
        print(f"[Joining] Joining room: {ROOM}")
        sio.emit('join_as_python_client', {'room': ROOM})
        print("[Listening] Waiting for sensor data...")
        print("-" * 80)

    # Disconnection event handler
    @sio.event
    def disconnect():
        print("\n[Disconnected] Connection closed")

    # Error handler
    @sio.event
    def connect_error(data):
        print(f"[Error] Connection error: {data}")

    @sio.event
    def error(data):
        print(f"[Error] {data.get('message', 'Unknown error')}")

    # Sensor data handler
    @sio.event
    def sensor_data(data):
        """Receive sensor data, flatten it, and print to console"""

        # Flatten the nested structure
        flattened = flatten_sensor_data(data)

        # Pretty print the flattened payload
        print(json.dumps(flattened, indent=2))
        print("-" * 80)

    # Connect to server
    try:
        print(f"[Connecting] Attempting to connect to {SERVER_URL}...")
        sio.connect(SERVER_URL, transports=['websocket', 'polling'])

        # Keep the connection alive
        sio.wait()

    except KeyboardInterrupt:
        print("\n[Interrupted] Shutting down...")
        sio.disconnect()
    except Exception as e:
        print(f"[Error] Failed to connect: {e}")


if __name__ == '__main__':
    print("=" * 80)
    print("Crowd Source - Flatten Sensor Data Example")
    print("=" * 80)
    print()
    main()
