#!/bin/bash

# Configuration
SERVER_USER="monolith"
SERVER_HOST="31.97.59.222"
REMOTE_PATH="/home/monolith/live/modules/crowd-source/source/client"
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
LOCAL_PATH="${SCRIPT_DIR}/client"
ZIP_NAME="client.zip"

# Colors for output
GREEN='\033[0;32m'
BLUE='\033[0;34m'
RED='\033[0;31m'
NC='\033[0m' # No Color

echo -e "${BLUE}=== Crowd Source Client Sync & Run ===${NC}"

# Step 1: Create zip on server
echo -e "${GREEN}[1/5] Creating zip on server...${NC}"
ssh ${SERVER_USER}@${SERVER_HOST} "cd ${REMOTE_PATH} && zip -r ${ZIP_NAME} *.py requirements.txt utils/ widgets/ 2>/dev/null || zip -r ${ZIP_NAME} *.py requirements.txt"

if [ $? -ne 0 ]; then
    echo -e "${RED}Error: Failed to create zip on server${NC}"
    exit 1
fi

# Step 2: Remove old local version
echo -e "${GREEN}[2/5] Removing old local version...${NC}"
rm -rf ${LOCAL_PATH}
mkdir -p ${LOCAL_PATH}

# Step 3: Download zip
echo -e "${GREEN}[3/5] Downloading client...${NC}"
scp ${SERVER_USER}@${SERVER_HOST}:${REMOTE_PATH}/${ZIP_NAME} ${LOCAL_PATH}/

if [ $? -ne 0 ]; then
    echo -e "${RED}Error: Failed to download client${NC}"
    exit 1
fi

# Step 4: Unzip
echo -e "${GREEN}[4/5] Extracting client...${NC}"
cd ${LOCAL_PATH}
unzip -q ${ZIP_NAME}
rm ${ZIP_NAME}

# Step 5: Install dependencies and run
echo -e "${GREEN}[5/5] Installing dependencies and running client...${NC}"

# Check if virtual environment exists, create if not
if [ ! -d "venv" ]; then
    echo "Creating virtual environment..."
    python3 -m venv venv
fi

# Activate virtual environment
source venv/bin/activate

# Install requirements
pip install -q -r requirements.txt

echo -e "${BLUE}=== Client Ready ===${NC}"
echo ""
echo "Two client versions available:"
echo "  1. CLI version:       python client.py --room <room> --server https://ws.vec4.net"
echo "  2. Textual UI (NEW):  python client_ui.py --room <room> --server https://ws.vec4.net"
echo ""

# Prompt user which version to run
read -p "Run with Textual UI? (Y=UI, n=CLI, s=skip): " -n 1 -r
echo ""

if [[ $REPLY =~ ^[Yy]$ ]] || [[ -z $REPLY ]]; then
    echo -e "${GREEN}Running Textual UI with default test room...${NC}"
    echo ""
    python client_ui.py --room test-org/test-room --server https://ws.vec4.net
elif [[ $REPLY =~ ^[Nn]$ ]]; then
    echo -e "${GREEN}Running CLI version with default test room...${NC}"
    echo ""
    python client.py --room test-org/test-room --server https://ws.vec4.net
else
    echo ""
    echo -e "${BLUE}To run manually:${NC}"
    echo "  cd ${LOCAL_PATH}"
    echo "  source venv/bin/activate"
    echo "  # Textual UI (recommended):"
    echo "  python client_ui.py --room test-org/test-room --server https://ws.vec4.net"
    echo "  # Or CLI version:"
    echo "  python client.py --room test-org/test-room --server https://ws.vec4.net"
    echo ""
fi

# Cleanup on server
echo ""
echo -e "${GREEN}Cleaning up server...${NC}"
ssh ${SERVER_USER}@${SERVER_HOST} "rm ${REMOTE_PATH}/${ZIP_NAME}"

echo -e "${BLUE}=== Done ===${NC}"
