"""Data formatting utilities for Crowd Source client"""

from datetime import datetime


def format_timestamp(timestamp_ms):
    """Format timestamp to readable format"""
    dt = datetime.fromtimestamp(timestamp_ms / 1000)
    return dt.strftime('%H:%M:%S.%f')[:-3]


def format_accelerometer_data(data):
    """Format accelerometer data for display"""
    return (
        f"Phone: {data['phoneId'][:8]}... | "
        f"X: {data['data']['x']:7.2f} | "
        f"Y: {data['data']['y']:7.2f} | "
        f"Z: {data['data']['z']:7.2f} | "
        f"Time: {format_timestamp(data['timestamp'])}"
    )


def format_accelerometer_rich(data):
    """Format accelerometer data with Rich markup"""
    phone_short = data['phoneId'][:8]
    x = data['data']['x']
    y = data['data']['y']
    z = data['data']['z']
    timestamp = format_timestamp(data['timestamp'])

    return (
        f"[cyan]{phone_short}[/cyan] | "
        f"X:[green]{x:7.2f}[/green] | "
        f"Y:[yellow]{y:7.2f}[/yellow] | "
        f"Z:[magenta]{z:7.2f}[/magenta] | "
        f"[dim]{timestamp}[/dim]"
    )


def format_gyroscope(gyro_data):
    """Format gyroscope data with abbreviated labels"""
    if not gyro_data:
        return None
    return {
        'alpha': f"{gyro_data['alpha']:6.2f}",
        'beta': f"{gyro_data['beta']:6.2f}",
        'gamma': f"{gyro_data['gamma']:6.2f}"
    }


def format_orientation(orient_data):
    """Format orientation data with abbreviated labels"""
    if not orient_data:
        return None
    return {
        'alpha': f"{orient_data['alpha']:6.1f}",
        'beta': f"{orient_data['beta']:6.1f}",
        'gamma': f"{orient_data['gamma']:6.1f}",
        'absolute': '✓' if orient_data.get('absolute') else '✗'
    }


def format_sensor_data(data):
    """Extract and format all sensor data from payload"""
    result = {
        'phone_id': data['phoneId'][:8],
        'timestamp': format_timestamp(data['timestamp'])
    }

    # Extract nested sensor data
    sensor_data = data.get('data', {})

    # Accelerometer (always present)
    if 'accelerometer' in sensor_data:
        accel = sensor_data['accelerometer']
        result['accel_x'] = f"{accel['x']:7.2f}"
        result['accel_y'] = f"{accel['y']:7.2f}"
        result['accel_z'] = f"{accel['z']:7.2f}"

    # Gyroscope (optional)
    if 'gyroscope' in sensor_data:
        gyro = sensor_data['gyroscope']
        result['gyro_alpha'] = f"{gyro['alpha']:6.2f}"
        result['gyro_beta'] = f"{gyro['beta']:6.2f}"
        result['gyro_gamma'] = f"{gyro['gamma']:6.2f}"

    # Orientation (optional)
    if 'orientation' in sensor_data:
        orient = sensor_data['orientation']
        result['orient_alpha'] = f"{orient['alpha']:6.1f}"
        result['orient_beta'] = f"{orient['beta']:6.1f}"
        result['orient_gamma'] = f"{orient['gamma']:6.1f}"
        result['orient_abs'] = '✓' if orient.get('absolute') else '✗'

    return result
