"""
OSC Sender

Low-level OSC message construction and UDP sending.
Wraps python-osc library with error handling and statistics tracking.
"""

from pythonosc.udp_client import SimpleUDPClient
from typing import Any, Optional


class OSCSender:
    """
    Handles low-level OSC message sending over UDP.

    Provides:
    - Enable/disable toggle
    - Message sending with error handling
    - Statistics tracking (message count, errors)
    - Dynamic target configuration
    """

    def __init__(self, host: str, port: int):
        """
        Initialize OSC sender.

        Args:
            host: Target IP address (e.g., "127.0.0.1")
            port: Target UDP port (e.g., 7000)
        """
        self.host = host
        self.port = port
        self.client = SimpleUDPClient(host, port)
        self.enabled = False

        # Statistics
        self.message_count = 0
        self.last_message = None
        self.last_error = None

    def send(self, address: str, *args: Any) -> bool:
        """
        Send OSC message to configured target.

        Args:
            address: OSC address string (e.g., "/crowd/sensor/all")
            *args: Variable arguments to send with message

        Returns:
            True if message sent successfully, False otherwise
        """
        if not self.enabled:
            return False

        try:
            # python-osc library requires a list, but we minimize allocations by:
            # 1. Passing args directly when possible
            # 2. Not storing args (only address) to avoid reference keeping
            if args:
                # Create minimal list (unavoidable - python-osc requirement)
                self.client.send_message(address, list(args))
            else:
                self.client.send_message(address, [])

            self.message_count += 1
            self.last_message = address  # Store address only (no args to reduce memory)

            return True

        except Exception as e:
            self.last_error = str(e)[:200]
            return False

    def enable(self):
        """Enable OSC message sending."""
        self.enabled = True

    def disable(self):
        """Disable OSC message sending."""
        self.enabled = False

    def update_target(self, host: str, port: int):
        """
        Update target host and port.

        Args:
            host: New target IP address
            port: New target UDP port
        """
        self.host = host
        self.port = port
        self.client = SimpleUDPClient(host, port)

    def get_stats(self) -> dict:
        """
        Get sender statistics.

        Returns:
            Dictionary with message_count, last_message, last_error
        """
        return {
            'message_count': self.message_count,
            'last_message': self.last_message,
            'last_error': self.last_error,
            'enabled': self.enabled,
            'target': f"{self.host}:{self.port}"
        }

    def reset_stats(self):
        """Reset message count and error tracking."""
        self.message_count = 0
        self.last_message = None
        self.last_error = None
