"""
OSC Message Validation and Conversion Utilities

Provides validation and sanitization functions for incoming OSC commands,
and conversion utilities for outbound data.
"""

from typing import Optional, Union


def validate_color_command(args: list) -> Optional[tuple]:
    """
    Validate RGB color command arguments.

    Supports two formats:
    - /crowd/page/color r g b (3 args)
    - /crowd/page/color/id deviceIdentifier r g b (4 args)

    Args:
        args: List of OSC arguments

    Returns:
        For 3 args: (r, g, b) with values clamped to 0.0-1.0
        For 4 args: (device_identifier, r, g, b) with RGB clamped
                    device_identifier can be phone ID (str) or index (int)
        None if invalid
    """
    try:
        if len(args) == 3:
            r = max(0.0, min(1.0, float(args[0])))
            g = max(0.0, min(1.0, float(args[1])))
            b = max(0.0, min(1.0, float(args[2])))
            return (r, g, b)
        elif len(args) == 4:
            device_identifier = args[0]  # Keep as-is (str or int)
            r = max(0.0, min(1.0, float(args[1])))
            g = max(0.0, min(1.0, float(args[2])))
            b = max(0.0, min(1.0, float(args[3])))
            return (device_identifier, r, g, b)
    except (ValueError, TypeError, IndexError):
        pass

    return None


def validate_frequency(hz: Union[int, float]) -> Optional[int]:
    """
    Validate sensor frequency in Hz and convert to milliseconds.

    Args:
        hz: Frequency in Hertz (1-60 recommended range)

    Returns:
        Milliseconds value clamped to server limits (16-5000ms), or None if invalid
    """
    try:
        hz = max(1, min(60, int(hz)))
        ms = int(1000 / hz)
        ms = max(16, min(5000, ms))  # Server enforces 16-5000ms range
        return ms
    except (ValueError, TypeError, ZeroDivisionError):
        return None


def sanitize_string(text: str, max_length: int = 200) -> str:
    """
    Sanitize string inputs for alerts and messages.

    Args:
        text: Input string
        max_length: Maximum allowed length

    Returns:
        Sanitized string truncated to max_length
    """
    return str(text)[:max_length]


def rgb_to_hex(r: float, g: float, b: float) -> str:
    """
    Convert RGB values (0.0-1.0 float range) to hex color string.

    Args:
        r: Red component (0.0-1.0)
        g: Green component (0.0-1.0)
        b: Blue component (0.0-1.0)

    Returns:
        Hex color string in format #RRGGBB
    """
    r_int = int(max(0.0, min(1.0, r)) * 255)
    g_int = int(max(0.0, min(1.0, g)) * 255)
    b_int = int(max(0.0, min(1.0, b)) * 255)
    return f"#{r_int:02x}{g_int:02x}{b_int:02x}"


def validate_phone_id(phone_id: str, connected_phones: set) -> bool:
    """
    Check if phone ID exists in connected phones set.

    Args:
        phone_id: Phone identifier to validate
        connected_phones: Set of currently connected phone IDs

    Returns:
        True if phone_id exists, False otherwise
    """
    return phone_id in connected_phones


def validate_page_path(page_path: str) -> bool:
    """
    Validate page path against known page types.

    Args:
        page_path: Page path/type to validate

    Returns:
        True if valid page type
    """
    valid_pages = ['init', 'page0', 'pageColour', 'pageMultiBtn']
    return page_path in valid_pages


def validate_osc_address(address: str) -> bool:
    """
    Validate OSC address against whitelist.

    Only allows addresses under /crowd/* namespace for security.

    Args:
        address: OSC address string

    Returns:
        True if address is allowed
    """
    return address.startswith('/crowd/')


def resolve_device_identifier(identifier, phone_index_map: dict) -> Optional[str]:
    """
    Resolve device identifier to phone ID.

    Accepts both phone ID (string) and phone index (integer) and resolves
    to the actual phone ID string for socket commands.

    Args:
        identifier: Either phone ID (str) or phone index (int)
        phone_index_map: Map of phoneId -> index

    Returns:
        Phone ID string, or None if not found

    Examples:
        >>> resolve_device_identifier("phone_abc123", {...})
        "phone_abc123"  # String ID passed through

        >>> resolve_device_identifier(1, {"phone_abc123": 1, "phone_def456": 2})
        "phone_abc123"  # Index 1 resolved to phone ID

        >>> resolve_device_identifier(999, {"phone_abc123": 1})
        None  # Invalid index
    """
    if isinstance(identifier, str):
        # Already a phone ID - verify it exists
        return identifier if identifier in phone_index_map else None

    elif isinstance(identifier, int):
        # Index - need reverse lookup
        for phone_id, idx in phone_index_map.items():
            if idx == identifier:
                return phone_id
        return None

    return None
