# Processor Architecture

## Overview

The processor architecture provides a scalable, efficient framework for real-time signal processing of sensor data from phones. Each processor can be independently enabled/disabled and configured through the OSC tab UI.

## Key Features

- ✅ **Zero overhead when disabled** - Early-exit checks prevent any processing when processors are off
- ✅ **Self-contained processors** - Each processor manages its own state, parameters, and buffers
- ✅ **UI auto-generation** - Processor parameters automatically appear in OSC tab
- ✅ **Hot-swappable** - Enable/disable processors at runtime without restarting
- ✅ **Error isolation** - Processor errors don't crash the system
- ✅ **Scalable** - Easy to add new processors

## Architecture Diagram

```
Sensor Data (Socket.IO)
         ↓
  send_sensor_data()
         ↓
  _needs_sensor_processing() ← CRITICAL EARLY-EXIT
         ↓                      (Returns False = zero overhead)
         ├─→ Raw Streams (if enabled)
         │   ├─ /crowd/sensor/all
         │   ├─ /crowd/sensor/accel
         │   ├─ /crowd/sensor/gyro
         │   └─ /crowd/sensor/orientation
         │
         └─→ Processors (if enabled)
             ├─ HitDetectorProcessor → /crowd/hit
             ├─ [Future: SmootherProcessor]
             └─ [Future: ScalerProcessor]
```

## How It Works

### 1. Early-Exit Optimization

**Problem:** Processing sensor data (extracting values, running algorithms) wastes CPU when nothing is enabled.

**Solution:** Check if ANY processing is needed BEFORE doing ANY work:

```python
def send_sensor_data(self, data: dict, phone_index: int = None):
    # CRITICAL: Early exit if nothing needs processing
    if not self._needs_sensor_processing():
        return  # ← ZERO overhead when all streams + processors disabled

    # Only extract data if we got past the early exit
    ...
```

### 2. Processor Registry Pattern

All processors live in a registry dict:

```python
self.processors = {
    'hit_detector': HitDetectorProcessor(enabled=False),
    # Add more processors here
}
```

**Benefits:**
- Centralized management
- Easy iteration (for proc in processors.values())
- Simple enable/disable (processors['hit_detector'].enabled = True)

### 3. Base Processor Class

All processors inherit from `Processor` base class:

```python
class Processor:
    def process(data, phone_id, phone_index, timestamp) -> dict | None:
        """Process data, return result or None"""

    def get_ui_config() -> dict:
        """Define UI controls for OSC tab"""

    def set_param(key, value):
        """Update parameter"""

    def reset():
        """Reset state (buffers, counters)"""

    def get_stats() -> dict:
        """Return metrics for monitoring"""
```

### 4. UI Auto-Generation

Processors define their own UI via `get_ui_config()`:

```python
def get_ui_config(self):
    return {
        'name': 'Hit Detector',
        'description': 'Detects hits from accelerometer peaks',
        'output_address': '/crowd/hit',
        'params': [
            {
                'key': 'buffer_length',
                'label': 'Buffer Length',
                'type': 'int',
                'default': 50,
                'min': 10,
                'max': 200,
                'hint': 'Number of samples to analyze'
            },
            ...
        ]
    }
```

The OSC tab UI is then manually created based on this config (could be auto-generated in the future).

## Hit Detector Processor

### Algorithm

1. **Buffer** recent Z-axis accelerometer values per device
2. **Calculate** peak (max) and baseline (median) from buffer
3. **Detect** hit when `(peak - baseline) > threshold`
4. **Enforce** refractory period (prevent double-triggers)
5. **Emit** hit event with strength value
6. **Clear** buffer to detect next hit

### Parameters

| Parameter | Type | Default | Range | Description |
|-----------|------|---------|-------|-------------|
| buffer_length | int | 50 | 10-200 | Number of samples to analyze |
| stride | int | 1 | 1-10 | Process every Nth sample (reduces CPU) |
| min_threshold | float | 10.0 | 1.0-50.0 | Minimum delta to trigger hit (m/s²) |
| refractory | float | 0.05 | 0.01-1.0 | Minimum seconds between hits |
| use_numpy | bool | True | - | Use NumPy for faster computation |
| debug | bool | False | - | Print debug messages to console |

### Output

OSC Message: `/crowd/hit <identifier> <timestamp> <strength>`

Example: `/crowd/hit 1 1761262244505 25.34`

- **identifier**: Phone ID or index (based on "Use Device Index" toggle)
- **timestamp**: Unix timestamp in milliseconds
- **strength**: Raw delta (peak - baseline) in m/s²

**Note:** Strength is NOT scaled to MIDI range. Use a downstream ScalerProcessor for that.

### Usage Example

1. **Enable processor:** Check "Enable Hit Detector" in OSC tab → Processors section
2. **Configure parameters:** Adjust buffer length, threshold, etc. and click "Apply Parameters"
3. **Tap phone:** Hit detector will emit `/crowd/hit` messages when it detects peaks
4. **Monitor stats:** Watch "Hits" and "Active Devices" counters update in real-time
5. **Reset state:** Click "Reset State" to clear all buffers and counters

### Performance

- **With NumPy:** ~0.1-0.2ms per device per sample (recommended)
- **Without NumPy:** ~0.3-0.5ms per device per sample (pure Python median calculation)
- **At 20Hz with 10 devices:** ~2-5ms total per update (negligible)

Processing happens inline in the Socket.IO thread. No worker threads needed unless you add very heavy processors (>5ms).

## Adding a New Processor

### Step 1: Create Processor File

Create `/client/utils/processors/your_processor.py`:

```python
from .base import Processor

class YourProcessor(Processor):
    def __init__(self, name="Your Processor", enabled=False):
        super().__init__(name=name, enabled=enabled)

        self.params = {
            'param1': 10,
            'param2': 0.5,
        }

    def get_ui_config(self):
        return {
            'name': self.name,
            'description': 'What your processor does',
            'output_address': '/crowd/your_output',
            'params': [
                {
                    'key': 'param1',
                    'label': 'Parameter 1',
                    'type': 'int',
                    'default': 10,
                    'hint': 'What this parameter controls'
                },
                ...
            ]
        }

    def process(self, data, phone_id, phone_index, timestamp):
        # Early exit if disabled
        if not self.enabled:
            return None

        # Your processing logic here
        result = do_something(data)

        if result:
            return {
                'device': phone_id,
                'value': result,
                'timestamp': timestamp
            }

        return None
```

### Step 2: Register in OSCService

Edit `/client/utils/osc_service.py`:

```python
# Add import
from utils.processors import HitDetectorProcessor, YourProcessor

# Add to registry in __init__
self.processors = {
    'hit_detector': HitDetectorProcessor(enabled=False),
    'your_processor': YourProcessor(enabled=False),  # ← Add this
}
```

### Step 3: Add Output Handler

In `/client/utils/osc_service.py`, add handler in `_send_processor_output()`:

```python
def _send_processor_output(self, proc_id: str, result: dict):
    # ... existing code ...

    elif proc_id == 'your_processor':
        device_id = result.get('device')
        value = result.get('value')
        timestamp = result.get('timestamp')

        identifier = device_index if self.app.osc_use_device_index else device_id

        self.sender.send('/crowd/your_output', identifier, timestamp, value)
```

### Step 4: Add UI to OSC Tab

Edit `/client/widgets/osc_tab.py` in the `compose()` method:

```python
# In Processors section, after Hit Detector
yield Label("Your Processor:", classes="processor-category")
yield Checkbox("Enable Your Processor (/crowd/your_output)",
               id="proc_your_processor_enabled", value=False)

with Container(classes="processor-params"):
    yield Label("  Parameter 1:", classes="param-label")
    yield Input(value="10", id="proc_your_processor_param1",
                classes="param-input")
    # ... more parameters ...

with Horizontal(classes="processor-actions"):
    yield Button("Apply Parameters", id="proc_your_processor_apply",
                 variant="primary")
    yield Button("Reset State", id="proc_your_processor_reset",
                 variant="warning")
```

### Step 5: Wire Up Event Handlers

In `/client/widgets/osc_tab.py`:

```python
# In on_button_pressed()
elif button_id == "proc_your_processor_apply":
    self._apply_your_processor_params()
elif button_id == "proc_your_processor_reset":
    self.app.osc_service.reset_processor('your_processor')

# In on_checkbox_changed()
elif checkbox_id == "proc_your_processor_enabled":
    self.app.osc_service.set_processor_enabled('your_processor', event.value)

# Add method
def _apply_your_processor_params(self):
    param1 = int(self.query_one("#proc_your_processor_param1", Input).value)
    self.app.osc_service.set_processor_param('your_processor', 'param1', param1)
    # ... more parameters ...
```

Done! Your processor is now fully integrated.

## Best Practices

### 1. Always Implement Early-Exit

```python
def process(self, data, ...):
    if not self.enabled:
        return None  # ← Zero overhead

    # Extract data only if enabled
    ...
```

### 2. Lazy Resource Allocation

Don't create buffers until needed:

```python
def _get_buffer(self, device):
    if device not in self._buffers:
        self._buffers[device] = deque(maxlen=self.params['buffer_length'])
    return self._buffers[device]
```

### 3. Type Conversion in on_param_change

```python
def on_param_change(self, key, value):
    if key == 'buffer_length':
        self.params[key] = max(10, min(200, int(value)))  # Clamp to range
```

### 4. Error Handling

OSCService wraps processor calls in try/except, but you should still handle edge cases:

```python
def process(self, data, ...):
    accel = data.get('accelerometer')
    if not accel:
        return None  # ← Gracefully handle missing data
```

### 5. Provide Useful Stats

```python
def get_stats(self):
    return {
        'enabled': self.enabled,
        'total_events': self._event_count,
        'active_devices': len(self._buffers),
        'buffer_usage': sum(len(b) for b in self._buffers.values())
    }
```

## Threading Considerations

### Current: Inline Processing (Socket Thread)

Processors currently run in the Socket.IO receive thread:

```
Socket.IO Event → send_sensor_data() → Processors → OSC Send
(all in same thread)
```

**Pros:**
- Simple, no threading complexity
- Low latency (~1-5ms)
- Fine for lightweight processors

**Cons:**
- Heavy processors block Socket.IO thread
- Could delay sensor data reception

### Future: Worker Thread Pool (If Needed)

If you add heavy processors (>5ms):

```python
class OSCService:
    def __init__(self):
        self.processor_queue = queue.Queue(maxsize=100)
        self.processor_thread = threading.Thread(
            target=self._processor_worker,
            daemon=True
        )
        self.processor_thread.start()

    def send_sensor_data(self, data):
        if self._has_heavy_processors():
            try:
                self.processor_queue.put_nowait((data, phone_index))
            except queue.Full:
                pass  # Drop if queue full
        else:
            self._process_inline(data, phone_index)
```

**Rule of thumb:** Profile first, optimize later. Start inline, move to threads only if >1ms latency measured.

## Troubleshooting

### Processor not receiving data

1. Check "Enable OSC Sending" is checked
2. Check processor is enabled
3. Check sensor streams are coming in (look at Status section)
4. Enable "Debug Output" to see console logs

### Parameters not applying

1. Make sure to click "Apply Parameters" button
2. Check console for error messages
3. Verify parameter values are in valid range

### Poor performance

1. Increase stride (process every Nth sample)
2. Reduce buffer_length
3. Disable debug output
4. Enable NumPy (if available)

### Hits not detected

1. Lower min_threshold
2. Shake phone harder
3. Enable debug output to see peak/baseline values
4. Check refractory period isn't too long

## Future Enhancements

- [ ] Auto-generate UI from `get_ui_config()` (currently manual)
- [ ] Processor chaining (pipe output of one to input of another)
- [ ] Processor presets (save/load parameter sets)
- [ ] Per-processor enable/disable via OSC commands
- [ ] Processor performance profiling built-in
- [ ] Worker thread pool for heavy processors
- [ ] NumPy-accelerated batch processing mode

## License

Part of the Crowd Source project.
