"""
Base Processor Class

All processors inherit from this base class which provides:
- Enable/disable functionality
- Parameter management
- UI configuration definition
- Common interface for processing
"""

from typing import Dict, List, Any, Optional


class Processor:
    """
    Base class for all signal processors.

    Architecture:
    - Each processor is self-contained with its own state and parameters
    - Enabled flag controls whether process() is called
    - UI configuration is defined by get_ui_config()
    - Zero overhead when disabled (early return before any computation)
    """

    def __init__(self, name: str, enabled: bool = False):
        """
        Initialize processor.

        Args:
            name: Human-readable processor name
            enabled: Whether processor starts enabled
        """
        self.name = name
        self.enabled = enabled
        self.params = {}

    def process(self, data: Dict[str, Any], phone_id: str = None,
                phone_index: int = None, timestamp: int = None) -> Optional[Dict[str, Any]]:
        """
        Process sensor data and return results.

        This method should be overridden by subclasses.

        Args:
            data: Sensor data dict with keys like:
                  {'accelerometer': {'x': 0.1, 'y': 0.2, 'z': 9.8},
                   'gyroscope': {'alpha': 0, 'beta': 0, 'gamma': 0},
                   'orientation': {...}}
            phone_id: Phone identifier (UUID string)
            phone_index: Phone index (integer, 0-based)
            timestamp: Unix timestamp in milliseconds

        Returns:
            Dict with processed results, or None if no output
            Example: {'hit_strength': 25.5, 'device': phone_id}
        """
        raise NotImplementedError("Subclasses must implement process()")

    def get_ui_config(self) -> Dict[str, Any]:
        """
        Return UI configuration for OSC tab.

        Returns:
            Dict with structure:
            {
                'name': 'Human Readable Name',
                'description': 'What this processor does',
                'output_address': '/crowd/processor_name',
                'params': [
                    {
                        'key': 'param_name',
                        'label': 'Display Label',
                        'type': 'int' | 'float' | 'bool' | 'string',
                        'default': <value>,
                        'min': <min_value>,  # optional, for numeric types
                        'max': <max_value>,  # optional, for numeric types
                        'step': <step>,      # optional, for numeric types
                        'hint': 'Help text'  # optional
                    },
                    ...
                ]
            }
        """
        return {
            'name': self.name,
            'description': 'Base processor',
            'output_address': f'/crowd/{self.name.lower().replace(" ", "_")}',
            'params': []
        }

    def set_param(self, key: str, value: Any) -> bool:
        """
        Update a parameter value.

        Args:
            key: Parameter key
            value: New value

        Returns:
            True if parameter was updated, False if key doesn't exist
        """
        if key in self.params:
            self.params[key] = value
            self.on_param_change(key, value)
            return True
        return False

    def on_param_change(self, key: str, value: Any):
        """
        Called when a parameter is changed.

        Override this to perform validation, type conversion, or
        trigger re-initialization when parameters change.

        Args:
            key: Parameter key that changed
            value: New value
        """
        pass

    def reset(self):
        """
        Reset processor state (buffers, counters, etc.).

        Override this if your processor maintains state.
        """
        pass

    def get_stats(self) -> Dict[str, Any]:
        """
        Return processor statistics for debugging/monitoring.

        Override this to expose internal metrics.

        Returns:
            Dict with stats like hit counts, buffer sizes, etc.
        """
        return {
            'enabled': self.enabled,
            'name': self.name
        }
