"""Socket.IO client connection management"""

import socketio
import time
from typing import Callable, Optional


class SocketClient:
    """Manages Socket.IO connection and event handlers"""

    def __init__(self, server_url: str, room: str):
        self.server_url = server_url
        self.room = room
        self.sio = socketio.Client()
        self.message_count = 0
        self.is_connected = False

        # Clock synchronization (NTP-style)
        self.clock_offset = 0  # offset = server_time - client_time (in ms)
        self.clock_sync_complete = False

        # Event callbacks
        self.on_connect_callback: Optional[Callable] = None
        self.on_disconnect_callback: Optional[Callable] = None
        self.on_data_callback: Optional[Callable] = None
        self.on_error_callback: Optional[Callable] = None
        self.on_phone_joined_callback: Optional[Callable] = None
        self.on_phone_left_callback: Optional[Callable] = None
        self.on_interaction_callback: Optional[Callable] = None
        self.on_clock_sync_callback: Optional[Callable] = None

        # Register internal handlers
        self._register_handlers()

    def _register_handlers(self):
        """Register Socket.IO event handlers"""

        @self.sio.event
        def connect():
            self.is_connected = True

            # LOG WHICH TRANSPORT IS BEING USED (for Windows latency diagnostics)
            transport = self.sio.transport()
            print(f"[SOCKET.IO] Connected using transport: {transport}")

            # Perform clock synchronization BEFORE joining room
            print(f"[CLOCK SYNC] Requesting time synchronization...")
            t1 = time.time() * 1000  # Client send time
            self.sio.emit('clock_sync_request', {'t1': t1})

            # NOTE: Room join happens in clock_sync_response handler

        @self.sio.event
        def disconnect():
            self.is_connected = False
            if self.on_disconnect_callback:
                self.on_disconnect_callback()

        @self.sio.event
        def connect_error(data):
            if self.on_error_callback:
                self.on_error_callback(f"Connection error: {data}")

        @self.sio.event
        def error(data):
            if self.on_error_callback:
                self.on_error_callback(data.get('message', 'Unknown error'))

        @self.sio.event
        def sensor_data(data):
            # CRITICAL: No timing capture, no dict modifications (reduces allocations)
            self.message_count += 1
            if self.on_data_callback:
                self.on_data_callback(data)

        @self.sio.on('phone_joined')
        def on_phone_joined(data):
            """Handle phone joined event"""
            if self.on_phone_joined_callback:
                self.on_phone_joined_callback(data)

        @self.sio.on('phone_left')
        def on_phone_left(data):
            """Handle phone left event"""
            if self.on_phone_left_callback:
                self.on_phone_left_callback(data)

        @self.sio.on('interaction')
        def on_interaction(data):
            """Handle button clicks and interactions"""
            if self.on_interaction_callback:
                self.on_interaction_callback(data)

        @self.sio.on('clock_sync_response')
        def on_clock_sync_response(data):
            """Handle clock sync response from server (NTP-style)"""
            t1 = data['t1']  # Client send time
            t2 = data['t2']  # Server receive time
            t3 = data['t3']  # Server send time
            t4 = time.time() * 1000  # Client receive time

            # Calculate offset using NTP algorithm
            # offset = ((server_time - client_time) on send + (server_time - client_time) on receive) / 2
            self.clock_offset = ((t2 - t1) + (t3 - t4)) / 2
            self.clock_sync_complete = True

            print(f"[CLOCK SYNC] Offset calculated: {self.clock_offset:.2f}ms")

            # Notify callback
            if self.on_clock_sync_callback:
                self.on_clock_sync_callback(self.clock_offset)

            # Notify connect callback
            if self.on_connect_callback:
                self.on_connect_callback()

            # NOW join room after sync is complete
            self.sio.emit('join_as_python_client', {'room': self.room})

    def connect(self):
        """Connect to Socket.IO server"""
        try:
            self.sio.connect(self.server_url, transports=['websocket', 'polling'])
            return True
        except Exception as e:
            if self.on_error_callback:
                self.on_error_callback(f"Failed to connect: {e}")
            return False

    def disconnect(self):
        """Disconnect from Socket.IO server"""
        if self.sio.connected:
            self.sio.disconnect()

    def request_phone_list(self) -> bool:
        """
        Request current phone list from server.

        Triggers server to send phone_joined events for all currently
        connected phones. Useful after clearing local phone list.

        Returns:
            True if request sent successfully, False otherwise
        """
        if not self.is_connected:
            return False

        try:
            # Re-emit join_as_python_client to trigger phone list refresh
            self.sio.emit('join_as_python_client', {'room': self.room})
            return True
        except Exception as e:
            if self.on_error_callback:
                self.on_error_callback(f"Failed to request phone list: {e}")
            return False

    def send_control_message(self, message_type: str, message_text: str):
        """Send a control message to all phones in the room"""
        if not self.is_connected:
            if self.on_error_callback:
                self.on_error_callback("Not connected to server")
            return False

        try:
            import time
            payload = {
                'type': message_type,
                'message': message_text,
                'timestamp': int(time.time() * 1000)
            }
            self.sio.emit('control_message', payload)
            return True
        except Exception as e:
            if self.on_error_callback:
                self.on_error_callback(f"Failed to send message: {e}")
            return False

    def navigate_to_page(self, page: str, phone_id: Optional[list] = None,
                         config: Optional[dict] = None) -> bool:
        """
        Navigate phone(s) to a specific page

        Args:
            page: Page type ('init', 'page0', 'pageColour', 'pageMultiBtn')
            phone_id: Optional list of phone IDs for targeting
            config: Optional page configuration dict

        Returns:
            True if sent successfully, False otherwise
        """
        if not self.is_connected:
            if self.on_error_callback:
                self.on_error_callback("Not connected to server")
            return False

        try:
            payload = {'page': page}
            if phone_id is not None:
                payload['phoneId'] = phone_id
            if config is not None:
                payload['config'] = config

            self.sio.emit('navigate_to_page', payload)
            return True
        except Exception as e:
            if self.on_error_callback:
                self.on_error_callback(f"Failed to navigate to page: {e}")
            return False

    def configure_page(self, page: str, config: dict,
                       phone_id: Optional[list] = None) -> bool:
        """
        Configure a page's appearance

        Args:
            page: Page type to configure
            config: Configuration dict
            phone_id: Optional list of phone IDs for targeting

        Returns:
            True if sent successfully, False otherwise
        """
        if not self.is_connected:
            if self.on_error_callback:
                self.on_error_callback("Not connected to server")
            return False

        try:
            payload = {'page': page, 'config': config}
            if phone_id is not None:
                payload['phoneId'] = phone_id

            self.sio.emit('configure_page', payload)
            return True
        except Exception as e:
            if self.on_error_callback:
                self.on_error_callback(f"Failed to configure page: {e}")
            return False

    def set_on_connect(self, callback: Callable):
        """Set callback for connection event"""
        self.on_connect_callback = callback

    def set_on_disconnect(self, callback: Callable):
        """Set callback for disconnection event"""
        self.on_disconnect_callback = callback

    def set_on_data(self, callback: Callable):
        """Set callback for accelerometer data event"""
        self.on_data_callback = callback

    def set_on_error(self, callback: Callable):
        """Set callback for error events"""
        self.on_error_callback = callback

    def set_on_phone_joined(self, callback: Callable):
        """Set callback for phone joined events"""
        self.on_phone_joined_callback = callback

    def set_on_phone_left(self, callback: Callable):
        """Set callback for phone left events"""
        self.on_phone_left_callback = callback

    def set_on_interaction(self, callback: Callable):
        """Set callback for interaction events"""
        self.on_interaction_callback = callback

    def set_on_clock_sync(self, callback: Callable):
        """Set callback for clock sync completion"""
        self.on_clock_sync_callback = callback

    def get_server_time(self, client_time_ms: float) -> float:
        """
        Convert client timestamp to server time using calculated offset.

        Args:
            client_time_ms: Client timestamp in milliseconds

        Returns:
            Adjusted timestamp in server time
        """
        return client_time_ms + self.clock_offset
