"""Config tab widget for page configuration"""

from typing import Optional, List, Dict
from textual.app import ComposeResult
from textual.containers import Horizontal, Vertical, VerticalScroll
from textual.widgets import Input, Button, Static, Label, Checkbox, TabbedContent, TabPane
from textual.message import Message
from textual.reactive import reactive


class ConfigTab(Vertical):
    """Config tab for configuring page appearance"""

    # Config memory (persists between navigations)
    pagecolour_config: reactive[Dict] = reactive({
        "pageInstanceName": "",
        "color": "#000000",
        "text": "",
        "textColor": "#FFFFFF"
    })
    multibtn_config: reactive[Dict] = reactive({
        'pageInstanceName': '',
        'mode': 'momentary',
        'backgroundColor': '#000000',
        'displayText': '',
        'displayTextColor': '#FFFFFF',
        'displayTextSize': 'medium',
        'buttonColor': '#FFFFFF',
        'buttonTextColor': '#000000',
        'buttons': [
            {'text': 'Option A'},
            {'text': 'Option B'}
        ],
        'submissionMessage': 'Thank you for your response!',
        'requireConfirmDialog': True
    })

    # Messages
    class NavigateToPage(Message):
        """Message sent when navigating to a page"""
        def __init__(self, page: str, phone_ids: Optional[List[str]], config: Optional[Dict]):
            super().__init__()
            self.page = page
            self.phone_ids = phone_ids
            self.config = config

    class ConfigurePage(Message):
        """Message sent when configuring a page"""
        def __init__(self, page: str, config: Dict, phone_ids: Optional[List[str]]):
            super().__init__()
            self.page = page
            self.config = config
            self.phone_ids = phone_ids

    def compose(self) -> ComposeResult:
        """Create config tab widgets with sub-tabs for each page type"""
        with TabbedContent():
            # PageColour sub-tab
            with TabPane("Solid Colour", id="tab-pagecolour"):
                with VerticalScroll():
                    yield Static("Solid Colour Configuration", classes="section-title")
                    with Vertical(id="pagecolour-config", classes="config-section"):
                        yield Static("Instance Name (required for OSC):")
                        yield Input(placeholder="e.g., color_screen_1", id="pagecolour-instance-name", value="color1")
                        yield Static("Color (hex):")
                        yield Input(placeholder="#000000", id="pagecolour-color")
                        yield Static("Text (optional overlay):")
                        yield Input(placeholder="", id="pagecolour-text")
                        yield Static("Text Color (hex):")
                        yield Input(placeholder="#FFFFFF", id="pagecolour-text-color")

                        yield Checkbox("Show calibration button", id="pagecolour-show-calibration", value=False)

                        with Horizontal(classes="config-action-buttons"):
                            yield Button("Apply Config", id="pagecolour-apply-config", variant="success")
                            yield Button("Apply & Navigate", id="pagecolour-apply-navigate", variant="primary")

            # Multi-Button sub-tab
            with TabPane("Multi-Button", id="tab-multibtn"):
                with VerticalScroll():
                    yield Static("Multi-Button Page Configuration", classes="section-title")
                    with Vertical(id="multibtn-config", classes="config-section"):
                        yield Static("Instance Name (required for OSC):")
                        yield Input(placeholder="e.g., vote_question_1", id="multibtn-instance-name", value="vote1")

                        yield Static("Question:")
                        yield Input(placeholder="Choose an option", id="multibtn-display-text")

                        yield Static("Interaction Mode:")
                        with Horizontal(id="multibtn-mode-container", classes="mode-buttons"):
                            yield Button("Momentary", id="multibtn-mode-momentary", variant="primary")
                            yield Button("Toggle", id="multibtn-mode-toggle", variant="default")
                            yield Button("Confirm", id="multibtn-mode-confirm", variant="default")

                        yield Static("Buttons (enable 1-5):", classes="subsection-label")

                        # Button 1
                        with Horizontal(classes="multibtn-button-row"):
                            yield Checkbox("Button 1", id="multibtn-btn-0-enabled", value=True)
                            yield Input(placeholder="Button text", id="multibtn-btn-0-text", value="Option A")

                        # Button 2
                        with Horizontal(classes="multibtn-button-row"):
                            yield Checkbox("Button 2", id="multibtn-btn-1-enabled", value=True)
                            yield Input(placeholder="Button text", id="multibtn-btn-1-text", value="Option B")

                        # Button 3
                        with Horizontal(classes="multibtn-button-row"):
                            yield Checkbox("Button 3", id="multibtn-btn-2-enabled", value=False)
                            yield Input(placeholder="Button text", id="multibtn-btn-2-text")

                        # Button 4
                        with Horizontal(classes="multibtn-button-row"):
                            yield Checkbox("Button 4", id="multibtn-btn-3-enabled", value=False)
                            yield Input(placeholder="Button text", id="multibtn-btn-3-text")

                        # Button 5
                        with Horizontal(classes="multibtn-button-row"):
                            yield Checkbox("Button 5", id="multibtn-btn-4-enabled", value=False)
                            yield Input(placeholder="Button text", id="multibtn-btn-4-text")

                        with Horizontal(classes="config-action-buttons"):
                            yield Button("Apply Config", id="multibtn-apply-config", variant="success")
                            yield Button("Apply & Navigate", id="multibtn-apply-navigate", variant="primary")

                        # Spacing
                        yield Static("")
                        yield Static("")

                        # Advanced Configuration Section
                        yield Static("Advanced Configuration", classes="subsection-label")
                        yield Static("")

                        yield Static("Question Size:")
                        with Horizontal(id="multibtn-text-size-container", classes="mode-buttons"):
                            yield Button("Small", id="multibtn-size-small", variant="default")
                            yield Button("Medium", id="multibtn-size-medium", variant="primary")
                            yield Button("Large", id="multibtn-size-large", variant="default")

                        yield Static("Submission Message:")
                        yield Input(placeholder="Thank you!", id="multibtn-submission-message")

                        yield Checkbox("Require confirmation dialog (confirm mode only)", id="multibtn-require-confirm-dialog", value=True)

                        yield Static("")

                        yield Static("Background Color (hex):")
                        yield Input(placeholder="#000000", id="multibtn-bg-color")

                        yield Static("Display Text Color (hex):")
                        yield Input(placeholder="#FFFFFF", id="multibtn-display-text-color")

                        yield Static("Global Button Color (hex):")
                        yield Input(placeholder="#FFFFFF", id="multibtn-button-color")

                        yield Static("Global Button Text Color (hex):")
                        yield Input(placeholder="#000000", id="multibtn-button-text-color")

    def on_button_pressed(self, event: Button.Pressed) -> None:
        """Handle all button clicks"""
        button_id = event.button.id

        # PageColour buttons
        if button_id == "pagecolour-apply-config":
            self._apply_config("pageColour")
        elif button_id == "pagecolour-apply-navigate":
            self._apply_and_navigate("pageColour")

        # Multi-button mode selection
        elif button_id == "multibtn-mode-momentary":
            self._set_multibtn_mode('momentary')
        elif button_id == "multibtn-mode-toggle":
            self._set_multibtn_mode('toggle')
        elif button_id == "multibtn-mode-confirm":
            self._set_multibtn_mode('confirm')

        # Multi-button text size
        elif button_id == "multibtn-size-small":
            self._set_multibtn_text_size('small')
        elif button_id == "multibtn-size-medium":
            self._set_multibtn_text_size('medium')
        elif button_id == "multibtn-size-large":
            self._set_multibtn_text_size('large')

        # Multi-button apply actions
        elif button_id == "multibtn-apply-config":
            self._apply_config("pageMultiBtn")
        elif button_id == "multibtn-apply-navigate":
            self._apply_and_navigate("pageMultiBtn")

    def _get_target_phone_ids(self) -> Optional[List[str]]:
        """Get phone IDs based on target mode from app state"""
        target_mode = self.app.target_mode

        if target_mode == "all":
            return None
        else:
            # Get selected phones from app state
            return list(self.app.selected_phone_ids)

    def _validate_instance_name(self, instance_name: str, page_type: str) -> bool:
        """
        Validate instance name format.

        Rules:
        - Required (not empty)
        - Only alphanumeric characters (a-z, A-Z, 0-9)
        - No spaces, hyphens, underscores, or special characters

        Args:
            instance_name: The instance name to validate
            page_type: Display name for error message (e.g., "Page Colour")

        Returns:
            True if valid, False otherwise (with notification shown)
        """
        # Check if empty
        if not instance_name:
            self.app.notify(
                "Instance name is required",
                title=f"{page_type} Validation Error",
                severity="error",
                timeout=5
            )
            return False

        # Check if alphanumeric only (no spaces, hyphens, underscores, special chars)
        if not instance_name.isalnum():
            # Build helpful error message
            invalid_chars = [c for c in instance_name if not c.isalnum()]
            unique_invalid = list(set(invalid_chars))

            self.app.notify(
                f"Instance name '{instance_name}' contains invalid characters: {', '.join(repr(c) for c in unique_invalid)}\n"
                f"Only letters (a-z, A-Z) and numbers (0-9) allowed.\n"
                f"No spaces, hyphens, underscores, or special characters.",
                title=f"{page_type} Validation Error",
                severity="error",
                timeout=8
            )
            return False

        return True

    def _build_pagecolour_config(self) -> Dict:
        """Build config dict from PageColour form inputs"""
        config = {}

        # Instance name is required
        instance_name = self.query_one("#pagecolour-instance-name", Input).value.strip()
        if not self._validate_instance_name(instance_name, "Solid Colour"):
            return {}  # Return empty if validation fails
        config["pageInstanceName"] = instance_name

        color = self.query_one("#pagecolour-color", Input).value.strip()
        if color:
            config["color"] = color

        text = self.query_one("#pagecolour-text", Input).value.strip()
        if text:
            config["text"] = text

        text_color = self.query_one("#pagecolour-text-color", Input).value.strip()
        if text_color:
            config["textColor"] = text_color

        show_calibration = self.query_one("#pagecolour-show-calibration", Checkbox).value
        config["showCalibration"] = show_calibration

        # Save to memory
        self.pagecolour_config = config
        return config

    def _apply_config(self, page: str) -> None:
        """Apply configuration without navigating"""
        if page == "pageColour":
            config = self._build_pagecolour_config()
        elif page == "pageMultiBtn":
            config = self._build_multibtn_config()
        else:
            return

        phone_ids = self._get_target_phone_ids()

        if config:
            self.post_message(self.ConfigurePage(page, config, phone_ids))

    def _apply_and_navigate(self, page: str) -> None:
        """Apply configuration and navigate phones to page"""
        if page == "pageColour":
            config = self._build_pagecolour_config()
        elif page == "pageMultiBtn":
            config = self._build_multibtn_config()
        else:
            return

        phone_ids = self._get_target_phone_ids()

        if config:
            self.post_message(self.NavigateToPage(page, phone_ids, config))

    def _set_multibtn_mode(self, mode: str):
        """Update mode and button variants"""
        config = dict(self.multibtn_config)
        config['mode'] = mode
        self.multibtn_config = config

        # Update button variants
        self.query_one("#multibtn-mode-momentary", Button).variant = "primary" if mode == "momentary" else "default"
        self.query_one("#multibtn-mode-toggle", Button).variant = "primary" if mode == "toggle" else "default"
        self.query_one("#multibtn-mode-confirm", Button).variant = "primary" if mode == "confirm" else "default"

    def _set_multibtn_text_size(self, size: str):
        """Update text size and button variants"""
        config = dict(self.multibtn_config)
        config['displayTextSize'] = size
        self.multibtn_config = config

        # Update button variants
        self.query_one("#multibtn-size-small", Button).variant = "primary" if size == "small" else "default"
        self.query_one("#multibtn-size-medium", Button).variant = "primary" if size == "medium" else "default"
        self.query_one("#multibtn-size-large", Button).variant = "primary" if size == "large" else "default"

    def _build_multibtn_config(self) -> Dict:
        """Build config dict from Multi-Button form inputs"""
        config = {}

        # Instance name is required
        instance_name = self.query_one("#multibtn-instance-name", Input).value.strip()
        if not self._validate_instance_name(instance_name, "Multi-Button"):
            return {}  # Return empty if validation fails
        config['pageInstanceName'] = instance_name

        # Mode (already in state)
        config['mode'] = self.multibtn_config['mode']

        # Background
        bg_color = self.query_one("#multibtn-bg-color", Input).value.strip()
        if bg_color:
            config['backgroundColor'] = bg_color

        # Display text
        display_text = self.query_one("#multibtn-display-text", Input).value.strip()
        if display_text:
            config['displayText'] = display_text

        display_text_color = self.query_one("#multibtn-display-text-color", Input).value.strip()
        if display_text_color:
            config['displayTextColor'] = display_text_color

        # Text size (already in state)
        config['displayTextSize'] = self.multibtn_config['displayTextSize']

        # Global button styles
        button_color = self.query_one("#multibtn-button-color", Input).value.strip()
        if button_color:
            config['buttonColor'] = button_color

        button_text_color = self.query_one("#multibtn-button-text-color", Input).value.strip()
        if button_text_color:
            config['buttonTextColor'] = button_text_color

        # Buttons (only include enabled ones)
        buttons = []
        for i in range(5):
            # Check if this button is enabled
            enabled = self.query_one(f"#multibtn-btn-{i}-enabled", Checkbox).value
            if not enabled:
                continue

            text = self.query_one(f"#multibtn-btn-{i}-text", Input).value.strip()

            if not text:
                continue  # Skip if no text provided

            button_config = {'text': text}
            buttons.append(button_config)

        if len(buttons) == 0:
            return {}  # Invalid: no buttons enabled

        config['buttons'] = buttons

        # Submission message
        submission_msg = self.query_one("#multibtn-submission-message", Input).value.strip()
        if submission_msg:
            config['submissionMessage'] = submission_msg

        # Require confirmation dialog
        require_confirm = self.query_one("#multibtn-require-confirm-dialog", Checkbox).value
        config['requireConfirmDialog'] = require_confirm

        # Save to memory
        self.multibtn_config = config
        return config
