"""Control panel widget for sending alerts and controlling frequency"""

from textual.app import ComposeResult
from textual.containers import Horizontal, Vertical
from textual.widgets import Input, Button, Static
from textual.message import Message
from textual.validation import Integer


class ControlPanel(Static):
    """Control panel for alerts and frequency control"""

    class SendAlert(Message):
        """Message sent when alert button is clicked"""
        def __init__(self, message: str):
            super().__init__()
            self.message = message

    class SetFrequency(Message):
        """Message sent when frequency is changed"""
        def __init__(self, frequency_ms: int):
            super().__init__()
            self.frequency_ms = frequency_ms

    def compose(self) -> ComposeResult:
        """Create control panel widgets"""
        # Alert controls
        yield Static("Send Alert to Phones", classes="section-title")
        with Horizontal(id="alert-controls"):
            yield Input(
                placeholder="Type alert message...",
                id="alert-input"
            )
            yield Button("Send", id="send-alert-btn", variant="primary")

        # Frequency controls
        yield Static("Set Streaming Frequency", classes="section-title")
        with Horizontal(id="frequency-controls"):
            yield Input(
                placeholder="50",
                id="frequency-input",
                type="integer"
            )
            yield Static("ms", id="frequency-unit")
            yield Button("Apply", id="apply-freq-btn", variant="success")

        # Frequency presets
        yield Static("Quick Presets:", id="preset-label")
        with Horizontal(id="frequency-presets"):
            yield Button("1000ms (1Hz)", id="preset-1000", variant="default")
            yield Button("200ms (5Hz)", id="preset-200", variant="default")
            yield Button("50ms (20Hz)", id="preset-50", variant="default")
            yield Button("20ms (50Hz)", id="preset-20", variant="default")

    def on_button_pressed(self, event: Button.Pressed) -> None:
        """Handle button presses"""
        if event.button.id == "send-alert-btn":
            self._send_alert()
        elif event.button.id == "apply-freq-btn":
            self._apply_frequency()
        elif event.button.id and event.button.id.startswith("preset-"):
            # Extract frequency from button id (e.g., "preset-50" -> 50)
            frequency = int(event.button.id.split("-")[1])
            self._set_frequency(frequency)

    def on_input_submitted(self, event: Input.Submitted) -> None:
        """Handle Enter key in inputs"""
        if event.input.id == "alert-input":
            self._send_alert()
        elif event.input.id == "frequency-input":
            self._apply_frequency()

    def _send_alert(self) -> None:
        """Send alert message and clear input"""
        alert_input = self.query_one("#alert-input", Input)
        message = alert_input.value.strip()
        if message:
            self.post_message(self.SendAlert(message))
            alert_input.value = ""
            alert_input.focus()

    def _apply_frequency(self) -> None:
        """Apply frequency from input field"""
        freq_input = self.query_one("#frequency-input", Input)
        try:
            frequency = int(freq_input.value)
            self._set_frequency(frequency)
        except ValueError:
            pass  # Invalid input, ignore

    def _set_frequency(self, frequency_ms: int) -> None:
        """Set frequency and update input display"""
        if 16 <= frequency_ms <= 5000:
            self.post_message(self.SetFrequency(frequency_ms))
            # Update input to show current value
            freq_input = self.query_one("#frequency-input", Input)
            freq_input.value = str(frequency_ms)
