"""Controls tab widget for page navigation and settings"""

from typing import Optional, List, Dict
from textual.app import ComposeResult
from textual.containers import Horizontal, VerticalScroll
from textual.widgets import Input, Button, Static, RadioButton, RadioSet
from textual.message import Message


class ControlsTab(VerticalScroll):
    """Controls tab for page navigation, targeting, frequency, and alerts"""

    # Messages
    class NavigateToPage(Message):
        """Message sent when navigating to a page"""
        def __init__(self, page: str, phone_ids: Optional[List[str]], config: Optional[Dict]):
            super().__init__()
            self.page = page
            self.phone_ids = phone_ids
            self.config = config

    class SetFrequency(Message):
        """Message sent when frequency is changed"""
        def __init__(self, frequency_ms: int):
            super().__init__()
            self.frequency_ms = frequency_ms

    class SendAlert(Message):
        """Message sent when alert button is clicked"""
        def __init__(self, message: str):
            super().__init__()
            self.message = message

    def compose(self) -> ComposeResult:
        """Create controls tab widgets"""
        # Page Navigation Section
        yield Static("Page Navigation", classes="section-title")
        yield Static("Quick Navigate:", classes="subsection-label")

        with Horizontal(classes="nav-buttons"):
            yield Button("INIT", id="nav-init", variant="primary")
            yield Button("PAGE0", id="nav-page0", variant="primary")
            yield Button("MULTIBTN", id="nav-multibtn", variant="primary")
            yield Button("COLOUR", id="nav-colour", variant="primary")

        yield Static("Target:", classes="subsection-label")
        with RadioSet(id="target-radio"):
            yield RadioButton("All Phones", id="target-all", value=True)
            yield RadioButton("Selected Phones", id="target-selected")

        # Frequency Control Section
        yield Static("Set Streaming Frequency", classes="section-title")
        with Horizontal(id="frequency-controls"):
            yield Input(
                placeholder="50",
                id="frequency-input",
                type="integer"
            )
            yield Static("ms", id="frequency-unit")
            yield Button("Apply", id="apply-freq-btn", variant="success")

        yield Static("Quick Presets:", id="preset-label")
        with Horizontal(id="frequency-presets"):
            yield Button("1000ms (1Hz)", id="preset-1000", variant="default")
            yield Button("200ms (5Hz)", id="preset-200", variant="default")
            yield Button("50ms (20Hz)", id="preset-50", variant="default")
            yield Button("20ms (50Hz)", id="preset-20", variant="default")

        # Alert Control Section
        yield Static("Send Alert to Phones", classes="section-title")
        with Horizontal(id="alert-controls"):
            yield Input(
                placeholder="Type alert message...",
                id="alert-input"
            )
            yield Button("Send", id="send-alert-btn", variant="primary")

    def on_radio_set_changed(self, event: RadioSet.Changed) -> None:
        """Handle target selection change"""
        if event.pressed.id == "target-all":
            self.app.target_mode = "all"
        elif event.pressed.id == "target-selected":
            self.app.target_mode = "selected"

    def on_button_pressed(self, event: Button.Pressed) -> None:
        """Handle all button clicks"""
        button_id = event.button.id

        # Quick navigation buttons
        if button_id == "nav-init":
            self._quick_navigate("init")
        elif button_id == "nav-page0":
            self._quick_navigate("page0")
        elif button_id == "nav-multibtn":
            self._quick_navigate("pageMultiBtn")
        elif button_id == "nav-colour":
            self._quick_navigate("pageColour")

        # Frequency controls
        elif button_id == "apply-freq-btn":
            self._apply_frequency()
        elif button_id and button_id.startswith("preset-"):
            frequency = int(button_id.split("-")[1])
            self._set_frequency(frequency)

        # Alert controls
        elif button_id == "send-alert-btn":
            self._send_alert()

    def on_input_submitted(self, event: Input.Submitted) -> None:
        """Handle Enter key in inputs"""
        if event.input.id == "alert-input":
            self._send_alert()
        elif event.input.id == "frequency-input":
            self._apply_frequency()

    def _quick_navigate(self, page: str) -> None:
        """Quick navigate to a page without config"""
        phone_ids = self._get_target_phone_ids()
        self.post_message(self.NavigateToPage(page, phone_ids, None))

    def _get_target_phone_ids(self) -> Optional[List[str]]:
        """Get phone IDs based on target mode from app state"""
        if self.app.target_mode == "all":
            return None
        else:
            # Get selected phones from app state
            return list(self.app.selected_phone_ids)

    def _apply_frequency(self) -> None:
        """Apply frequency from input field"""
        freq_input = self.query_one("#frequency-input", Input)
        try:
            frequency = int(freq_input.value)
            self._set_frequency(frequency)
        except ValueError:
            pass

    def _set_frequency(self, frequency_ms: int) -> None:
        """Set frequency and update input display"""
        if 16 <= frequency_ms <= 5000:
            self.post_message(self.SetFrequency(frequency_ms))
            freq_input = self.query_one("#frequency-input", Input)
            freq_input.value = str(frequency_ms)

    def _send_alert(self) -> None:
        """Send alert message and clear input"""
        alert_input = self.query_one("#alert-input", Input)
        message = alert_input.value.strip()
        if message:
            self.post_message(self.SendAlert(message))
            alert_input.value = ""
            alert_input.focus()
