"""Data stream widget for displaying sensor data"""

from textual.widgets import DataTable
from textual.coordinate import Coordinate
from rich.text import Text


class DataStream(DataTable):
    """Live table display for sensor data - one row per device"""

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.device_rows = {}  # phoneId -> row_key mapping
        self.device_counts = {}  # phoneId -> message count
        self.columns_initialized = False
        self.has_gyro = False
        self.has_orientation = False
        self.cursor_type = "none"
        self.zebra_stripes = True

    def on_mount(self) -> None:
        """Set up initial table columns (accelerometer only)"""
        self.add_columns(
            "Phone",
            "Acc X",
            "Acc Y",
            "Acc Z",
            "Time",
            "Count"
        )
        self.columns_initialized = True

    def _add_gyro_columns(self):
        """Add gyroscope columns if not already added"""
        if not self.has_gyro:
            self.add_column("Gα°/s")
            self.add_column("Gβ°/s")
            self.add_column("Gγ°/s")
            self.has_gyro = True

    def _add_orientation_columns(self):
        """Add orientation columns if not already added"""
        if not self.has_orientation:
            self.add_column("Oα°")
            self.add_column("Oβ°")
            self.add_column("Oγ°")
            self.add_column("Abs")
            self.has_orientation = True

    def add_data(self, data: dict, message_num: int):
        """Add or update sensor data for a device"""
        phone_id = data['phoneId']
        phone_short = phone_id[:8]

        # Track per-device message count
        if phone_id not in self.device_counts:
            self.device_counts[phone_id] = 0
        self.device_counts[phone_id] += 1
        device_count = self.device_counts[phone_id]

        # Format timestamp
        from utils.formatters import format_timestamp
        time_str = format_timestamp(data['timestamp'])

        # Extract nested sensor data
        sensor_data = data.get('data', {})

        # Check if we need to add new columns
        has_gyro_data = 'gyroscope' in sensor_data
        has_orient_data = 'orientation' in sensor_data

        if has_gyro_data and not self.has_gyro:
            self._add_gyro_columns()

        if has_orient_data and not self.has_orientation:
            self._add_orientation_columns()

        # Build cell list
        cells = []

        # Phone ID
        cells.append(Text(phone_short, style="cyan"))

        # Accelerometer (always present)
        if 'accelerometer' in sensor_data:
            accel = sensor_data['accelerometer']
            cells.append(Text(f"{accel['x']:7.2f}", style="green"))
            cells.append(Text(f"{accel['y']:7.2f}", style="yellow"))
            cells.append(Text(f"{accel['z']:7.2f}", style="magenta"))
        else:
            cells.extend([Text("-"), Text("-"), Text("-")])

        # Time and Count
        cells.append(Text(time_str, style="dim"))
        cells.append(Text(str(device_count), style="dim"))

        # Gyroscope (if available)
        if self.has_gyro:
            if has_gyro_data:
                gyro = sensor_data['gyroscope']
                cells.append(Text(f"{gyro['alpha']:6.2f}", style="red"))
                cells.append(Text(f"{gyro['beta']:6.2f}", style="bright_red"))
                cells.append(Text(f"{gyro['gamma']:6.2f}", style="red"))
            else:
                cells.extend([Text("-"), Text("-"), Text("-")])

        # Orientation (if available)
        if self.has_orientation:
            if has_orient_data:
                orient = sensor_data['orientation']
                cells.append(Text(f"{orient['alpha']:6.1f}", style="cyan"))
                cells.append(Text(f"{orient['beta']:6.1f}", style="bright_cyan"))
                cells.append(Text(f"{orient['gamma']:6.1f}", style="cyan"))
                abs_symbol = "✓" if orient.get('absolute') else "✗"
                cells.append(Text(abs_symbol, style="dim"))
            else:
                cells.extend([Text("-"), Text("-"), Text("-"), Text("-")])

        # Update existing row or add new one
        if phone_id in self.device_rows:
            row_key = self.device_rows[phone_id]
            row_index = self.get_row_index(row_key)

            # Update each cell
            try:
                for col_index, cell in enumerate(cells):
                    self.update_cell_at(
                        Coordinate(row_index, col_index),
                        cell,
                        update_width=False
                    )
            except Exception as e:
                pass  # Ignore update errors
        else:
            # Add new row
            row_key = self.add_row(*cells)
            self.device_rows[phone_id] = row_key
