"""
OSC Tab Widget

Provides UI for OSC configuration, stream control, and monitoring.
"""

from textual.app import ComposeResult
from textual.widgets import Checkbox, Input, Button, Static, Label
from textual.containers import Container, Vertical, Horizontal, VerticalScroll


class OSCTab(VerticalScroll):
    """
    OSC configuration and monitoring tab.

    Features:
    - Enable/disable OSC sending
    - Configure send/receive ports and host
    - Toggle individual data streams
    - View real-time statistics
    - Monitor aggregation state
    """

    def compose(self) -> ComposeResult:
        """Compose OSC tab layout."""
        # Configuration Section
        yield Label("OSC Configuration", classes="section-header")
        with Container(classes="config-panel"):
            yield Checkbox("Enable OSC Sending", id="osc_enabled")
            yield Checkbox("Use Device Index (instead of ID)", id="osc_use_index")

            with Horizontal(classes="input-row"):
                yield Label("Send Host:", classes="input-label")
                yield Input(value="127.0.0.1", id="send_host", classes="input-field")

            with Horizontal(classes="input-row"):
                yield Label("Send Port:", classes="input-label")
                yield Input(value="7000", id="send_port", classes="input-field")

            with Horizontal(classes="input-row"):
                yield Label("Receive Port:", classes="input-label")
                yield Input(value="7001", id="receive_port", classes="input-field")

            with Horizontal(classes="button-row"):
                yield Button("Apply Configuration", id="apply_config", variant="primary")
                yield Button("Test Connection", id="test_connection")

        # Stream Toggles Section
        yield Label("Data Streams", classes="section-header")
        with Container(classes="streams-panel"):
            yield Label("Sensor Streams:", classes="stream-category")
            yield Checkbox("All Sensors Combined (/crowd/sensor/all)", id="stream_sensor_all", value=False)
            yield Checkbox("Accelerometer Only (/crowd/sensor/accel)", id="stream_sensor_accel", value=False)
            yield Checkbox("Gyroscope Only (/crowd/sensor/gyro)", id="stream_sensor_gyro", value=False)
            yield Checkbox("Orientation Only (/crowd/sensor/orientation)", id="stream_sensor_orientation", value=False)

            yield Label("Interaction Streams:", classes="stream-category")
            yield Checkbox("Button Interactions (/crowd/interaction)", id="stream_interaction", value=False)

            yield Label("Aggregated Data Streams:", classes="stream-category")
            yield Checkbox("Phone Button States (/crowd/state/phones/*)", id="stream_phone_states", value=False)
            yield Checkbox("Phone Names (/crowd/state/phones/names)", id="stream_phone_names", value=False)
            yield Checkbox("Vote Counts (/crowd/votes/*)", id="stream_votes", value=False)
            yield Checkbox("Total Press Counts (/crowd/totals/*)", id="stream_totals", value=False)

            yield Label("Connection Event Streams:", classes="stream-category")
            yield Checkbox("Phone Joined (/crowd/phone/joined)", id="stream_phone_joined", value=False)
            yield Checkbox("Phone Left (/crowd/phone/left)", id="stream_phone_left", value=False)
            yield Checkbox("Phone Count (/crowd/phone/count)", id="stream_phone_count", value=False)

        # Processors Section
        yield Label("Processors", classes="section-header")
        with Container(classes="processors-panel"):
            # Hit Detector Processor
            yield Label("Hit Detector:", classes="processor-category")
            yield Checkbox("Enable Hit Detector (/crowd/hit)", id="proc_hit_detector_enabled", value=False)

            with Container(classes="processor-params"):
                yield Label("  Buffer Length:", classes="param-label")
                yield Input(value="50", id="proc_hit_detector_buffer_length", classes="param-input")

                yield Label("  Stride:", classes="param-label")
                yield Input(value="1", id="proc_hit_detector_stride", classes="param-input")

                yield Label("  Min Threshold:", classes="param-label")
                yield Input(value="10.0", id="proc_hit_detector_min_threshold", classes="param-input")

                yield Label("  Refractory Period (s):", classes="param-label")
                yield Input(value="0.05", id="proc_hit_detector_refractory", classes="param-input")

                yield Checkbox("  Use NumPy", id="proc_hit_detector_use_numpy", value=True)
                yield Checkbox("  Debug Output", id="proc_hit_detector_debug", value=False)

            with Horizontal(classes="processor-actions"):
                yield Button("Apply Parameters", id="proc_hit_detector_apply", variant="primary")
                yield Button("Reset State", id="proc_hit_detector_reset", variant="warning")

            yield Static("Hits: 0 | Active Devices: 0", id="proc_hit_detector_stats", classes="processor-stats")

        # Status Section
        yield Label("Status", classes="section-header")
        with Container(classes="status-panel"):
            yield Static("Status: ○ Disconnected", id="status_connection")
            yield Static("Outbound: 0 messages", id="status_outbound")
            yield Static("Inbound: 0 messages", id="status_inbound")
            yield Static("Last Sent: -", id="status_last_sent")
            yield Static("Last Received: -", id="status_last_received")

        # Aggregation State Section
        yield Label("Aggregation State", classes="section-header")
        with Container(classes="aggregation-panel"):
            yield Static("Current Page: -", id="agg_page")
            yield Static("Active Phones: 0", id="agg_phones")
            yield Static("Button Votes: []", id="agg_votes")
            yield Button("Reset Aggregations", id="reset_agg", variant="error")

    def on_mount(self):
        """Load configuration from app state and start refresh timer."""
        app = self.app

        # Load config from app reactive state
        if hasattr(app, 'osc_enabled'):
            self.query_one("#osc_enabled", Checkbox).value = app.osc_enabled
        if hasattr(app, 'osc_use_device_index'):
            self.query_one("#osc_use_index", Checkbox).value = app.osc_use_device_index
        if hasattr(app, 'osc_send_host'):
            self.query_one("#send_host", Input).value = app.osc_send_host
        if hasattr(app, 'osc_send_port'):
            self.query_one("#send_port", Input).value = str(app.osc_send_port)
        if hasattr(app, 'osc_receive_port'):
            self.query_one("#receive_port", Input).value = str(app.osc_receive_port)

        # Start stats refresh timer (every second)
        self.set_interval(1.0, self.refresh_stats)

    def on_button_pressed(self, event: Button.Pressed):
        """Handle button clicks."""
        button_id = event.button.id

        if button_id == "apply_config":
            self._apply_config()
        elif button_id == "test_connection":
            self._test_connection()
        elif button_id == "reset_agg":
            if hasattr(self.app, 'osc_service'):
                self.app.osc_service.aggregator.reset()

        # Processor buttons
        elif button_id == "proc_hit_detector_apply":
            self._apply_hit_detector_params()
        elif button_id == "proc_hit_detector_reset":
            if hasattr(self.app, 'osc_service'):
                self.app.osc_service.reset_processor('hit_detector')
                self.app.notify("Hit Detector state reset", severity="information")

    def on_checkbox_changed(self, event: Checkbox.Changed):
        """Handle checkbox toggles."""
        checkbox_id = event.checkbox.id

        if checkbox_id == "osc_enabled":
            # Update app state
            if hasattr(self.app, 'osc_enabled'):
                self.app.osc_enabled = event.value

            # Enable/disable sender
            if hasattr(self.app, 'osc_service'):
                if event.value:
                    self.app.osc_service.enable()
                else:
                    self.app.osc_service.disable()

        elif checkbox_id == "osc_use_index":
            # Update app state
            if hasattr(self.app, 'osc_use_device_index'):
                self.app.osc_use_device_index = event.value

        # Stream toggles
        elif checkbox_id.startswith("stream_"):
            stream_name = checkbox_id.replace("stream_", "")
            if hasattr(self.app, 'osc_service'):
                self.app.osc_service.set_stream_enabled(stream_name, event.value)

        # Processor enable/disable
        elif checkbox_id == "proc_hit_detector_enabled":
            if hasattr(self.app, 'osc_service'):
                self.app.osc_service.set_processor_enabled('hit_detector', event.value)

        # Processor parameter checkboxes (use_numpy, debug)
        elif checkbox_id.startswith("proc_hit_detector_"):
            param_name = checkbox_id.replace("proc_hit_detector_", "")
            if param_name in ("use_numpy", "debug"):
                if hasattr(self.app, 'osc_service'):
                    self.app.osc_service.set_processor_param('hit_detector', param_name, event.value)

    def _apply_config(self):
        """Apply configuration changes."""
        try:
            send_host = self.query_one("#send_host", Input).value
            send_port = int(self.query_one("#send_port", Input).value)
            receive_port = int(self.query_one("#receive_port", Input).value)

            # Update app state
            if hasattr(self.app, 'osc_send_host'):
                self.app.osc_send_host = send_host
            if hasattr(self.app, 'osc_send_port'):
                self.app.osc_send_port = send_port
            if hasattr(self.app, 'osc_receive_port'):
                self.app.osc_receive_port = receive_port

            # Update service
            if hasattr(self.app, 'osc_service'):
                self.app.osc_service.update_config(send_host, send_port, receive_port)

        except ValueError:
            # Invalid port number
            pass

    def _test_connection(self):
        """Send test OSC message."""
        if hasattr(self.app, 'osc_service'):
            import time
            timestamp = int(time.time() * 1000)
            self.app.osc_service.sender.send('/crowd/test', 'hello', timestamp, 123.45)

    def _apply_hit_detector_params(self):
        """Apply hit detector parameters from UI."""
        if not hasattr(self.app, 'osc_service'):
            return

        try:
            # Get values from inputs
            buffer_length = int(self.query_one("#proc_hit_detector_buffer_length", Input).value)
            stride = int(self.query_one("#proc_hit_detector_stride", Input).value)
            min_threshold = float(self.query_one("#proc_hit_detector_min_threshold", Input).value)
            refractory = float(self.query_one("#proc_hit_detector_refractory", Input).value)

            # Apply parameters
            self.app.osc_service.set_processor_param('hit_detector', 'buffer_length', buffer_length)
            self.app.osc_service.set_processor_param('hit_detector', 'stride', stride)
            self.app.osc_service.set_processor_param('hit_detector', 'min_threshold', min_threshold)
            self.app.osc_service.set_processor_param('hit_detector', 'refractory', refractory)

            self.app.notify("Hit Detector parameters applied", severity="information")

        except ValueError as e:
            self.app.notify(f"Invalid parameter value: {e}", severity="error", timeout=5)

    def refresh_stats(self):
        """Update statistics display (called every second)."""
        if not hasattr(self.app, 'osc_service'):
            return

        stats = self.app.osc_service.get_stats()

        # Connection status
        status_text = "● Connected" if stats['enabled'] else "○ Disconnected"
        self.query_one("#status_connection", Static).update(f"Status: {status_text}")

        # Message counts
        self.query_one("#status_outbound", Static).update(f"Outbound: {stats['outbound_count']} messages")
        self.query_one("#status_inbound", Static).update(f"Inbound: {stats['inbound_count']} messages")

        # Last messages (truncate for display)
        last_sent = stats.get('last_sent') or '-'
        last_received = stats.get('last_received') or '-'
        self.query_one("#status_last_sent", Static).update(f"Last Sent: {str(last_sent)[:80]}")
        self.query_one("#status_last_received", Static).update(f"Last Received: {str(last_received)[:80]}")

        # Aggregation state
        page_id, phone_count, phone_ids, states = self.app.osc_service.aggregator.get_phone_states_data()
        _, _, _, vote_counts = self.app.osc_service.aggregator.get_votes_data()

        self.query_one("#agg_page", Static).update(f"Current Page: {page_id or '-'}")
        self.query_one("#agg_phones", Static).update(f"Active Phones: {phone_count}")
        self.query_one("#agg_votes", Static).update(f"Button Votes: {vote_counts}")

        # Processor stats
        processors = stats.get('processors', {})
        if 'hit_detector' in processors:
            hit_stats = processors['hit_detector']
            total_hits = hit_stats.get('total_hits', 0)
            active_devices = hit_stats.get('active_devices', 0)
            self.query_one("#proc_hit_detector_stats", Static).update(
                f"Hits: {total_hits} | Active Devices: {active_devices}"
            )

    DEFAULT_CSS = """
    .section-header {
        margin-top: 1;
        margin-bottom: 1;
        color: $accent;
        text-style: bold;
    }

    .config-panel, .streams-panel, .status-panel, .aggregation-panel, .processors-panel {
        height: auto;
        padding: 1;
        border: solid $primary;
        margin-bottom: 1;
    }

    .processor-category {
        margin-top: 1;
        margin-bottom: 1;
        color: $accent;
        text-style: bold;
    }

    .processor-params {
        height: auto;
        margin-left: 2;
        margin-top: 1;
        margin-bottom: 1;
    }

    .param-label {
        margin-bottom: 1;
        color: $text-muted;
    }

    .param-input {
        width: 20;
        margin-bottom: 1;
    }

    .processor-actions {
        height: auto;
        margin-top: 1;
        margin-bottom: 1;
        margin-left: 2;
    }

    .processor-stats {
        margin-left: 2;
        margin-top: 1;
        color: $success;
        text-style: italic;
    }

    .input-row {
        height: auto;
        align: left middle;
        margin-bottom: 1;
    }

    .input-label {
        width: 15;
        margin-right: 1;
    }

    .input-field {
        width: 30;
    }

    .button-row {
        height: auto;
        margin-top: 1;
    }

    .stream-category {
        margin-top: 1;
        margin-bottom: 1;
        color: $text;
        text-style: bold;
    }

    OSCTab Checkbox {
        margin-left: 2;
        margin-bottom: 1;
    }

    OSCTab Static {
        margin-bottom: 1;
    }

    OSCTab Button {
        margin-right: 1;
    }

    OSCTab Container {
        height: auto;
    }

    OSCTab Horizontal {
        height: auto;
    }
    """
