"""Phones tab widget for viewing and selecting connected phones"""

from textual.app import ComposeResult
from textual.containers import Horizontal, Vertical, VerticalScroll, Container
from textual.widgets import Static, Button
from textual.reactive import reactive
from rich.text import Text
import time


class PhoneItem(Static):
    """Individual clickable phone item"""

    def __init__(self, phone_id: str, index: int, connected_at: float, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.phone_id = phone_id
        self.index = index
        self.connected_at = connected_at

    def on_click(self, event) -> None:
        """Handle click to toggle selection"""
        # Toggle selection in app state
        phones_tab = self.parent.parent  # Container -> PhonesTab
        if phones_tab and hasattr(phones_tab, 'app'):
            selected = set(phones_tab.app.selected_phone_ids)
            if self.phone_id in selected:
                selected.discard(self.phone_id)
            else:
                selected.add(self.phone_id)
            phones_tab.app.selected_phone_ids = selected


class PhonesTab(VerticalScroll):
    """Phones tab for displaying and selecting connected phones"""

    def compose(self) -> ComposeResult:
        """Create phones tab widgets"""
        yield Static("Connected Phones (0)", id="phones-title", classes="section-title")

        # Index Assignment Info
        yield Static("Index Assignment", classes="subsection-label")
        yield Static(
            "[dim]Uses Lowest Available indexing (1, 2, 3...N).\n"
            "Indices are reused when phones disconnect.\n"
            "Example: Phones [1, 5, 12] → next phone gets index 2.\n\n"
            "State automatically syncs with server on connect/reconnect.[/dim]",
            classes="info-text"
        )

        yield Container(id="phones-container")
        with Horizontal(id="phone-controls"):
            yield Button("Select All", id="select-all-btn", variant="default")
            yield Button("Clear", id="clear-btn", variant="default")
        yield Static("Selected: 0 phone(s)", id="selection-count", classes="subsection-label")
        yield Static("Tip: Click directly on a phone to toggle its selection", classes="subsection-label")

    def on_mount(self) -> None:
        """Set up watchers for app state changes"""
        # Watch for changes to connected phones
        self.watch(self.app, "connected_phones", self._refresh_phone_list)
        self.watch(self.app, "selected_phone_ids", self._update_selection_display)

        # Set up periodic refresh for time updates (every 10 seconds)
        self.set_interval(10, self._auto_refresh_time)

    def _refresh_phone_list(self, phones: list) -> None:
        """Rebuild phone list when phones change"""
        container = self.query_one("#phones-container", Container)

        # Update title
        title = self.query_one("#phones-title", Static)
        title.update(f"Connected Phones ({len(phones)})")

        # Remove all existing phone items
        container.remove_children()

        if not phones:
            container.mount(Static("No phones connected", classes="empty-state"))
            return

        # Create individual clickable widgets for each phone
        for phone in phones:
            phone_id = phone['phone_id']
            index = phone['index']
            connected_at = phone['connected_at']

            # Calculate time since connection
            elapsed = time.time() - connected_at
            if elapsed < 60:
                time_str = f"{int(elapsed)}s ago"
            elif elapsed < 3600:
                time_str = f"{int(elapsed/60)}m ago"
            else:
                time_str = f"{int(elapsed/3600)}h ago"

            # Check if selected
            is_selected = phone_id in self.app.selected_phone_ids
            checkbox = "☑" if is_selected else "☐"

            # Format: [☑/☐] #N: phoneId (time)
            phone_display = f"{checkbox} #{index}: {phone_id[:8]} (connected {time_str})"

            # Create clickable phone item
            phone_item = PhoneItem(
                phone_id=phone_id,
                index=index,
                connected_at=connected_at,
                classes="phone-item"
            )
            phone_item.update(phone_display)
            container.mount(phone_item)

    def _update_selection_display(self, selected_ids: set) -> None:
        """Update the selection count display"""
        count_label = self.query_one("#selection-count", Static)
        count_label.update(f"Selected: {len(selected_ids)} phone(s)")

        # Refresh the list to update checkboxes
        self._refresh_phone_list(self.app.connected_phones)

    def _auto_refresh_time(self) -> None:
        """Periodically refresh to update 'connected X ago' times"""
        if self.app.connected_phones:
            self._refresh_phone_list(self.app.connected_phones)

    def on_button_pressed(self, event: Button.Pressed) -> None:
        """Handle button presses"""
        if event.button.id == "select-all-btn":
            self._select_all()
        elif event.button.id == "clear-btn":
            self._clear_selection()

    def _select_all(self) -> None:
        """Select all phones"""
        if self.app.connected_phones:
            self.app.selected_phone_ids = {p['phone_id'] for p in self.app.connected_phones}

    def _clear_selection(self) -> None:
        """Clear all selections"""
        self.app.selected_phone_ids = set()
