"""Status bar widget showing connection info"""

from textual.widgets import Static
from textual.reactive import reactive


class StatusBar(Static):
    """Display status information: room, connection, clock offset"""

    room = reactive("")
    is_connected = reactive(False)
    clock_offset = reactive(0.0)
    clock_synced = reactive(False)

    def watch_room(self, room: str) -> None:
        """Update display when room changes"""
        self._update_display()

    def watch_is_connected(self, connected: bool) -> None:
        """Update display when connection status changes"""
        self._update_display()

    def watch_clock_offset(self, offset: float) -> None:
        """Update display when clock offset changes"""
        self.clock_synced = True
        self._update_display()

    def _update_display(self) -> None:
        """Build and update the complete status line"""
        # Room
        room_text = f"Room: {self.room}"

        # Connection status
        if self.is_connected:
            status_text = "[green]Connected ✓[/green]"
        else:
            status_text = "[red]Disconnected ✗[/red]"

        # Clock offset
        if not self.clock_synced:
            clock_text = "Clock: --"
        elif abs(self.clock_offset) < 1:
            clock_text = f"[green]Clock: {self.clock_offset:.1f}ms[/green]"
        elif abs(self.clock_offset) < 10:
            clock_text = f"[cyan]Clock: {self.clock_offset:.1f}ms[/cyan]"
        elif abs(self.clock_offset) < 50:
            clock_text = f"[yellow]Clock: {self.clock_offset:.1f}ms[/yellow]"
        else:
            clock_text = f"[red]Clock: {self.clock_offset:.0f}ms[/red]"

        # Combine all parts with separators (NO message count)
        full_text = f"{room_text}  |  {status_text}  |  {clock_text}"
        self.update(full_text)
