"""Tabbed container widget with main tabs"""

from textual.app import ComposeResult
from textual.containers import Container
from textual.widgets import Static, TabbedContent, TabPane
from textual.message import Message

from widgets.controls_tab import ControlsTab
from widgets.config_tab import ConfigTab
from widgets.phones_tab import PhonesTab
from widgets.latency_diagnostic_tab import LatencyDiagnosticTab
from widgets.event_log_tab import EventLogTab
from widgets.osc_tab import OSCTab


class TabbedContainer(Static):
    """Main tabbed container with Controls, Page Builder, Phones, Latency Diagnostic, Event Log, and OSC tabs"""

    class TabChanged(Message):
        """Message sent when tab is switched"""
        def __init__(self, tab_id: str):
            super().__init__()
            self.tab_id = tab_id

    def compose(self) -> ComposeResult:
        """Create tabbed interface"""
        with TabbedContent(initial="controls"):
            with TabPane("Controls", id="controls"):
                yield ControlsTab(id="controls-tab")
            with TabPane("Page Builder", id="config"):
                yield ConfigTab(id="config-tab")
            with TabPane("Phones", id="phones"):
                yield PhonesTab(id="phones-tab")
            with TabPane("Latency", id="latency"):
                yield LatencyDiagnosticTab(id="latency-diagnostic-tab")
            with TabPane("Event Log", id="events"):
                yield EventLogTab(id="event-log-tab")
            with TabPane("OSC", id="osc"):
                yield OSCTab(id="osc-tab")

    def on_tabbed_content_tab_activated(self, event: TabbedContent.TabActivated) -> None:
        """Handle tab switching - pause/resume updates"""
        tab_id = event.pane.id

        # Pause all tabs first
        self._pause_all_tabs()

        # Resume the active tab
        if tab_id == "events":
            event_tab = self.query_one("#event-log-tab", EventLogTab)
            event_tab.resume()

        # Emit message to parent
        self.post_message(self.TabChanged(tab_id))

    def _pause_all_tabs(self) -> None:
        """Pause updates for all tabs"""
        try:
            event_tab = self.query_one("#event-log-tab", EventLogTab)
            event_tab.pause()
        except Exception:
            pass
